from typing import Callable, List

from maps_adv.adv_store.api.schemas.enums import (
    OverviewPositionEnum,
    PlatformEnum,
)
from maps_adv.export.lib.core.enum import (
    ActionType,
    CampaignType,
    CreativeType,
)

from .utils import (
    filter_campaign_by_overview_position,
    filter_campaign_by_platform,
    filter_campaign_by_type,
)

from . import feature


class Page:
    def __init__(
        self,
        name: str,
        features: List[feature.BaseFeature],
        filters: List[Callable] = tuple(),
    ):
        self.name = name
        self.features = features
        self.filters = filters


NAVIGATOR_PAGES = [
    Page(
        "navi_billboard",
        features=[
            feature.Init(),
            feature.UserDisplayLimits(3),
            feature.WithActionType(ActionType.SEARCH, 4),
            feature.HideAdvLabel(5),
            feature.WithCampaignType(CampaignType.PIN_ON_ROUTE, 6),
            feature.WithActionType(ActionType.OPEN_SITE, 7),
            feature.WithBigBillboard(8),
        ],
        filters=[
            filter_campaign_by_platform(PlatformEnum.NAVI),
            filter_campaign_by_type(CampaignType.PIN_ON_ROUTE, CampaignType.BILLBOARD),
        ],
    ),
    Page(
        "navi_zero_speed_banner",
        features=[
            feature.Init(),
            feature.WithActionType(ActionType.OPEN_SITE, 2),
            feature.WithCreativeType(CreativeType.AUDIO_BANNER, 3),
            feature.WithActionType(ActionType.RESOLVE_URI, 4),
            feature.MainAction(5),
            # Skipping 6 due to GEODISPLAY-147
            feature.WithAuditPixel(7),
            feature.WithActionType(ActionType.ADD_POINT_TO_ROUTE, 8),
            feature.WithAuditPixel(9, at_least=2),
            feature.WithTemplatelessDcm(10, for_testing_only=True),
        ],
        filters=[
            filter_campaign_by_platform(PlatformEnum.NAVI),
            filter_campaign_by_type(CampaignType.ZERO_SPEED_BANNER),
        ],
    ),
    Page(
        "route_via_point",
        features=[feature.Init()],
        filters=[
            filter_campaign_by_platform(PlatformEnum.NAVI),
            filter_campaign_by_type(CampaignType.ROUTE_VIA_POINT),
        ],
    ),
    Page(
        "overview_banner",
        features=[
            feature.Init(),
            feature.WithActionType(ActionType.RESOLVE_URI, 2),
            feature.ResolveUriBrowserTarget(3),
            feature.WithAuditPixel(4),
            feature.WithAuditPixel(5, at_least=2),
            feature.WithTemplatelessDcm(6, for_testing_only=True),
        ],
        filters=[
            filter_campaign_by_platform(PlatformEnum.NAVI),
            filter_campaign_by_type(CampaignType.OVERVIEW_BANNER),
            filter_campaign_by_overview_position(
                None, OverviewPositionEnum.START, OverviewPositionEnum.ALL
            ),
        ],
    ),
    Page(
        "overview_finish",
        features=[
            feature.Init(3),
            feature.WithAuditPixel(4),
            feature.WithAuditPixel(5, at_least=2),
            feature.WithTemplatelessDcm(6, for_testing_only=True),
        ],
        filters=[
            filter_campaign_by_platform(PlatformEnum.NAVI),
            filter_campaign_by_type(CampaignType.OVERVIEW_BANNER),
            filter_campaign_by_overview_position(
                OverviewPositionEnum.FINISH, OverviewPositionEnum.ALL
            ),
        ],
    ),
    Page(
        "navi_menu_icon",
        features=[feature.Init()],
        filters=[
            filter_campaign_by_platform(PlatformEnum.NAVI),
            filter_campaign_by_type(CampaignType.CATEGORY, CampaignType.PIN_SEARCH),
        ],
    ),
]

MOBILE_MAPS_PAGES = [
    Page(
        "mobile_maps_route_pins",
        features=[
            feature.Init(),
            feature.WithCampaignType(CampaignType.BILLBOARD, 2),
            feature.WithBigBillboard(2),
        ],
        filters=[
            filter_campaign_by_platform(PlatformEnum.MAPS),
            filter_campaign_by_type(CampaignType.PIN_ON_ROUTE, CampaignType.BILLBOARD),
        ],
    ),
    Page(
        "mobile_maps_menu_icon",
        features=[feature.Init()],
        filters=[
            filter_campaign_by_platform(PlatformEnum.MAPS),
            filter_campaign_by_type(CampaignType.CATEGORY, CampaignType.PIN_SEARCH),
        ],
    ),
]

METRO_PAGES = [
    Page(
        "metro_route_banner",
        features=[feature.Init()],
        filters=[
            filter_campaign_by_platform(PlatformEnum.METRO),
            filter_campaign_by_type(CampaignType.ROUTE_BANNER),
        ],
    ),
    Page(
        "metro_promocode",
        features=[feature.Init()],
        filters=[
            filter_campaign_by_platform(PlatformEnum.METRO),
            filter_campaign_by_type(CampaignType.PROMOCODE),
        ],
    ),
]
