from .segments import clients_with_segments_cte

create_client = """
    WITH new_client AS (
        INSERT INTO clients (
            biz_id,
            first_name,
            last_name,
            passport_uid,
            phone,
            email,
            gender,
            comment
        )
        VALUES ($3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING *
    )
    INSERT INTO client_revisions (
        client_id,
        source,
        metadata,
        biz_id,
        first_name,
        last_name,
        passport_uid,
        phone,
        email,
        gender,
        comment,
        initiator_id
    )
    SELECT id, $1, $2, biz_id, first_name, last_name, passport_uid,
        phone, email, gender, comment, $11
    FROM new_client
    RETURNING
        client_id AS id,
        biz_id,
        first_name,
        last_name,
        passport_uid,
        phone::bigint,
        email,
        gender,
        comment,
        source,
        created_at AS registration_timestamp
"""

find_client = """
    SELECT id, phone::bigint, email, passport_uid
    FROM clients
    WHERE biz_id=$1 AND ({filter_params_str})
    ORDER BY created_at
"""

retrieve_client = f"""
    WITH {clients_with_segments_cte}
    SELECT
        c.id,
        c.first_name,
        c.last_name,
        c.biz_id,
        c.passport_uid,
        c.gender,
        c.phone::bigint,
        c.email,
        c.comment,
        FIRST_VALUE(client_revisions.source) OVER (
            PARTITION BY client_revisions.client_id
            ORDER BY client_revisions.created_at
        ) AS source,
        c.labels,
        c.cleared_for_gdpr,
        c.created_at AS registration_timestamp,
        cws.segments,
        es.total_orders_created_count AS stat_order_total,
        es.total_orders_accepted_count AS stat_order_successful,
        es.total_orders_rejected_count AS stat_order_unsuccessful,
        es.last_order_created_ts AS stat_order_last_created_ts
    FROM clients AS c
    JOIN clients_with_segments AS cws ON c.id=cws.client_id
    JOIN events_stat AS es ON c.id=es.client_id
    JOIN client_revisions ON client_revisions.client_id = c.id
"""

client_exists = """
    SELECT EXISTS(
        SELECT 1
        FROM clients
        WHERE id=$1 AND biz_id=$2
    )
"""

list_suggest_clients = """
    SELECT id,
        biz_id,
        phone::bigint,
        email,
        first_name,
        last_name
    FROM clients
    WHERE biz_id=$1
        AND {search_field} ILIKE $2
    ORDER BY id DESC
    LIMIT $3
"""

list_contacts = """
    SELECT id,
        biz_id,
        passport_uid,
        phone::bigint,
        email,
        first_name,
        last_name,
        cleared_for_gdpr
    FROM clients
    WHERE id = ANY($1)
    ORDER BY created_at
"""

update_client = f"""
    WITH {clients_with_segments_cte},
    updated_client AS (
        UPDATE clients SET
        phone=$6,
        email=$7,
        passport_uid=$8,
        first_name=$9,
        last_name=$10,
        gender=$11,
        comment=$12
        WHERE biz_id=$1 AND id=$3
        RETURNING *
    ), new_revision AS (
        INSERT INTO client_revisions (
            client_id,
            biz_id,
            source,
            metadata,
            phone,
            email,
            passport_uid,
            first_name,
            last_name,
            gender,
            comment,
            initiator_id
        )
        SELECT id, biz_id, $4, $5, phone, email,
               passport_uid, first_name, last_name, gender, comment, $13
        FROM updated_client
        RETURNING *
    )
    SELECT
        new_revision.client_id AS id,
        new_revision.biz_id,
        new_revision.phone::bigint,
        new_revision.email,
        new_revision.passport_uid,
        new_revision.first_name,
        new_revision.last_name,
        new_revision.gender,
        new_revision.comment,
        FIRST_VALUE(client_revisions.source) OVER (
            PARTITION BY client_revisions.client_id
            ORDER BY client_revisions.created_at
        ) AS source,
        updated_client.labels,
        new_revision.created_at AS registration_timestamp,
        cws.segments,
        es.total_orders_created_count AS stat_order_total,
        es.total_orders_accepted_count AS stat_order_successful,
        es.total_orders_rejected_count AS stat_order_unsuccessful,
        es.last_order_created_ts AS stat_order_last_created_ts
    FROM new_revision
    JOIN updated_client ON updated_client.id = new_revision.client_id
    JOIN client_revisions ON client_revisions.client_id = new_revision.client_id
    JOIN clients_with_segments AS cws ON cws.client_id = new_revision.client_id
    JOIN events_stat AS es ON es.client_id = new_revision.client_id
"""

duplicate_client_exists = """
    SELECT EXISTS(
        SELECT id
        FROM clients
        WHERE biz_id=$1
            AND id != $2
            AND {passport_uid_clause}
            AND ({search_fields_clause})
    )
"""

merge_client = (
    f"WITH {clients_with_segments_cte},"
    + """
    merged_client AS (
        UPDATE clients
        -- we are ready for empty 'clients_set_string'
        SET biz_id=biz_id {clients_set_str}
        WHERE biz_id=$1 AND id=$3
        RETURNING *
    ), new_revision AS (
        INSERT INTO client_revisions (
            client_id,
            biz_id,
            source,
            metadata,
            initiator_id
            {revision_fields}
        )
        SELECT id, biz_id, $4, $5, $6 {revision_fields}
        FROM merged_client
        RETURNING *
    )
    SELECT
        new_revision.client_id AS id,
        new_revision.biz_id,
        new_revision.phone::bigint,
        new_revision.email,
        new_revision.passport_uid,
        new_revision.first_name,
        new_revision.last_name,
        new_revision.gender,
        new_revision.comment,
        FIRST_VALUE(client_revisions.source) OVER (
            PARTITION BY client_revisions.client_id
            ORDER BY client_revisions.created_at
        ) AS source,
        merged_client.labels,
        new_revision.created_at AS registration_timestamp,
        cws.segments,
        es.total_orders_created_count AS stat_order_total,
        es.total_orders_accepted_count AS stat_order_successful,
        es.total_orders_rejected_count AS stat_order_unsuccessful,
        es.last_order_created_ts AS stat_order_last_created_ts
    FROM new_revision
    JOIN merged_client ON merged_client.id = new_revision.client_id
    JOIN client_revisions ON client_revisions.client_id = new_revision.client_id
    JOIN clients_with_segments AS cws ON cws.client_id = new_revision.client_id
    JOIN events_stat AS es ON es.client_id = new_revision.client_id
"""
)


list_client_segments = f"""
    WITH {clients_with_segments_cte}
    SELECT client_id,
        biz_id,
        segments,
        labels
    FROM clients_with_segments
    ORDER BY client_id DESC
"""


clear_clients_by_passport = """
    WITH cleared_clients AS (
        UPDATE clients
        SET first_name = NULL,
            last_name = NULL,
            passport_uid = NULL,
            phone = NULL,
            email = NULL,
            gender = NULL,
            cleared_for_gdpr = true
        WHERE passport_uid = $1
        RETURNING id, biz_id
    ),
    cleared_revisions AS (
        UPDATE client_revisions
        SET first_name = NULL,
            last_name = NULL,
            passport_uid = NULL,
            phone = NULL,
            email = NULL,
            gender = NULL
        FROM cleared_clients
        WHERE client_id = cleared_clients.id
    ),
    new_revision AS (
        INSERT INTO client_revisions (
            client_id,
            biz_id,
            source
        )
        SELECT id, biz_id, $2
        FROM cleared_clients
    )
    SELECT id AS client_id, biz_id
    FROM cleared_clients
    ORDER BY id
"""


check_clients_existence_by_passport = """
    SELECT EXISTS(
        SELECT 1
        FROM clients
        WHERE passport_uid = $1
    )
"""
