from aiohttp.web import Request, Response

from maps_adv.common.lasagna import tvm_auth

from .providers import ApiProvider


class Resources:
    _provider: ApiProvider

    def __init__(self, provider: ApiProvider):
        self._provider = provider

    async def ping(self, _: Request) -> Response:
        return Response(status=204)

    async def generate_landing_data(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.generate_landing_data(data=body)

        return Response(status=201, body=result)

    async def create_landing_from_data(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.create_landing_from_data(data=body)

        return Response(status=201, body=result)

    @tvm_auth.only("TVM_DELETE_LANDING_WHITELIST")
    async def delete_landing(self, request: Request) -> Response:
        body = await request.read()

        await self._provider.delete_landing(data=body)

        return Response(status=204)

    @tvm_auth.skip
    async def fetch_published_landing_data(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.fetch_published_landing_data(data=body)

        return Response(status=200, body=result)

    @tvm_auth.skip
    async def get_landing_config(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.get_landing_config(data=body)

        return Response(status=200, body=result)

    async def check_slug_is_free(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.check_slug_is_free(data=body)

        return Response(status=200, body=result)

    async def update_slug(self, request: Request) -> Response:
        body = await request.read()

        await self._provider.update_slug(data=body)

        return Response(status=204)

    async def set_landing_publicity(self, request: Request) -> Response:
        body = await request.read()

        await self._provider.set_landing_publicity(data=body)

        return Response(status=204)

    async def set_blocked(self, request: Request) -> Response:
        body = await request.read()

        await self._provider.set_blocked(data=body)

        return Response(status=204)

    async def show_landing_details(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.show_landing_details(data=body)

        return Response(status=200, body=result)

    async def edit_landing_details(self, request: Request) -> Response:
        body = await request.read()

        result = await self._provider.edit_landing_details(data=body)

        return Response(status=200, body=result)

    async def suggest_color_presets(self, request: Request) -> Response:
        result = await self._provider.suggest_color_presets()

        return Response(status=200, body=result)

    async def suggest_field_values(self, request: Request) -> Response:
        field = request.match_info["field"]
        body = await request.read()

        result = await self._provider.suggest_field_values(data=body, field=field)

        return Response(status=200, body=result)

    async def edit_instagram_landing(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.edit_instagram_landing(data=body)
        return Response(status=200, body=result)

    async def fetch_landing_phone(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.fetch_landing_phone(data=body)
        return Response(status=200, body=result)

    async def fetch_landing_photos(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.fetch_landing_photos(data=body)
        return Response(status=200, body=result)

    async def hide_landing_photos(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.hide_landing_photos(data=body)
        return Response(status=200, body=result)

    async def get_landing_slug(self, request: Request) -> Response:
        body = await request.read()
        result = await self._provider.get_landing_slug(data=body)
        return Response(status=200, body=result)
