import logging
import telebot
from telebot.async_telebot import AsyncTeleBot

from maps_adv.geosmb.tuner.client import TunerClient
from maps_adv.geosmb.notification_bot.server.lib.bot.handlers.users import register_user_commands
from maps_adv.geosmb.notification_bot.server.lib.bot.utils.notify_admins import notify_admins
from maps_adv.geosmb.notification_bot.server.lib.bot.middlewares.antiflood_echo import AntifloodEcho


__all__ = ["Domain"]


telebot.logger.setLevel(logging.INFO)


class Domain:
    __slots__ = (
        "_bot",
        "_config",
        "_tuner_client"
    )

    _config: dict
    _bot: AsyncTeleBot
    _tuner_client: TunerClient

    def __init__(self, bot: AsyncTeleBot, tuner_client: TunerClient, config: dict):
        self._bot = bot
        self._tuner_client = tuner_client
        self._config = config

    async def send_message(self, user_id: int, message: str, disable_notification: bool = False):
        await self._bot.send_message(user_id, message, disable_notification=disable_notification)

    async def process_updates(self, update: dict):
        await self._bot.process_new_updates([update])

    async def handle_startup(self):
        register_user_commands(self._bot, self._tuner_client)
        self._bot.setup_middleware(AntifloodEcho(self._config["CUSTOM_RATE_LIMIT"], bot=self._bot))

        await self._bot.delete_webhook()
        await self._bot.set_webhook(url=self._config["WEBHOOK_URL"])

        await self._bot.set_my_commands(
            commands=[
                telebot.types.BotCommand("start", "Включить уведомления"),
                telebot.types.BotCommand("stop", "Отключить уведомления"),
                telebot.types.BotCommand("help", "Помощь")
            ],
        )
        await notify_admins(self._bot, self._config["BOT_ADMINS"])
