-- enums

CREATE TYPE event_source AS ENUM ('DESKTOP_MAPS', 'MOBILE', 'SPRAV');
CREATE TYPE event_type AS ENUM (
    'REVIEW',
    'CLICK_ON_PHONE',
    'OPEN_SITE',
    'MAKE_ROUTE',
    'VIEW_WORKING_HOURS',
    'VIEW_ENTRANCES',
    'CTA_BUTTON_CLICK',
    'FAVOURITE_CLICK',
    'LOCATION_SHARING',
    'BOOKING_SECTION_INTERACTION',
    'SHOWCASE_PRODUCT_CLICK',
    'PROMO_TO_SITE',
    'GEOPRODUCT_BUTTON_CLICK'
);
CREATE TYPE lead_source AS ENUM ('YT');

-- leads
CREATE TABLE leads
(
	id bigserial PRIMARY KEY,
	biz_id bigint NOT NULL,
	passport_uid varchar(20),
	device_id varchar(36),
	yandex_uid varchar(32),
	source lead_source NOT NULL,
	name varchar(256) NOT NULL,
	created_at timestamptz DEFAULT now() NOT NULL,
    CONSTRAINT ck_leads_name_not_empty_string
        CHECK (length((name)::text) > 0),
	CONSTRAINT uq_leads_biz_id_and_passport_uid
		UNIQUE (biz_id, passport_uid),
	CONSTRAINT ck_leads_id_fields_not_all_null CHECK (
		NOT (
		    (passport_uid IS NULL)
		    AND (device_id IS NULL)
		    AND (yandex_uid IS NULL)
		)
    )
);


CREATE INDEX leads_biz_id_btree_idx
    ON leads (biz_id);

CREATE INDEX leads_passport_uid_btree_idx
    ON leads (passport_uid);

CREATE INDEX leads_device_id_btree_idx
    ON leads (device_id);

CREATE INDEX leads_yandex_uid_btree_idx
    ON leads (yandex_uid);

CREATE INDEX leads_biz_id_device_id_btree_idx
    ON leads (biz_id, device_id);

CREATE INDEX leads_biz_id_yandex_uid_btree_idx
    ON leads (biz_id, yandex_uid);

CREATE INDEX leads_device_id_passport_uid_null_btree_idx
    ON leads (device_id)
    WHERE (passport_uid IS NULL);

CREATE INDEX leads_passport_uid_passport_uid_null_btree_idx
    ON leads (passport_uid)
    WHERE (passport_uid IS NULL);




-- lead_revisions

CREATE TABLE lead_revisions
(
    id bigserial PRIMARY KEY,
    lead_id bigint NOT NULL
        CONSTRAINT fk_lead_revisions_lead_id_leads
            REFERENCES leads,
    biz_id bigint NOT NULL,
    passport_uid varchar(20),
    device_id varchar(36),
    yandex_uid varchar(32),
    source lead_source NOT NULL,
    name varchar(256),
    created_at timestamptz DEFAULT now() NOT NULL
);

CREATE index lead_revisions_lead_id_btree_idx
    ON lead_revisions (lead_id);

CREATE index lead_revisions_biz_id_btree_idx
    ON lead_revisions (biz_id);

-- lead_events

CREATE TABLE lead_events
(
    id bigserial PRIMARY KEY,
    lead_id bigint NOT NULL
        CONSTRAINT fk_lead_events_lead_id_leads
            REFERENCES leads,
    biz_id bigint NOT NULL,
    event_type event_type NOT NULL,
    event_value varchar(64)
        CONSTRAINT ck_lead_events_lead_events_event_value_not_empty_string
            CHECK (length((event_value)::text) > 0),
    events_amount integer NOT NULL
        CONSTRAINT ck_lead_events_lead_events_events_amount_is_positive
            CHECK (events_amount > 0),
    event_timestamp timestamptz NOT NULL,
    source event_source NOT NULL,
    created_at timestamptz DEFAULT now() NOT NULL,
    CONSTRAINT ck_lead_review_must_be_with_score CHECK (
        (
            (event_value IS NOT NULL)
            AND (event_type = 'REVIEW'::event_type)
        )
        OR (event_type <> 'REVIEW'::event_type)
    )
);

CREATE index lead_events_event_value_btree_idx
    ON lead_events (event_value);

CREATE index lead_events_biz_id_btree_idx
    ON lead_events (biz_id);

CREATE index lead_events_event_type_btree_idx
    ON lead_events (event_type);

CREATE index lead_events_lead_id_btree_idx
    ON lead_events (lead_id);

CREATE index lead_events_biz_id_lead_id_btree_idx
    ON lead_events (biz_id, lead_id);

-- events_stat_precalced

CREATE TABLE events_stat_precalced
(
    id bigserial PRIMARY KEY,
    lead_id bigint NOT NULL,
    biz_id bigint NOT NULL,
    recent_total_count bigint NOT NULL,
    total_clicks_on_phone bigint NOT NULL,
    total_site_opens bigint NOT NULL,
    total_make_routes bigint NOT NULL,
    last_review_rating text,
    last_activity_timestamp timestamptz NOT NULL,
    total_view_working_hours bigint NOT NULL,
    total_view_entrances bigint NOT NULL,
    total_cta_button_click bigint NOT NULL,
    total_favourite_click bigint NOT NULL,
    total_location_sharing bigint NOT NULL,
    total_booking_section_interaction bigint NOT NULL,
    total_showcase_product_click bigint NOT NULL,
    total_promo_to_site bigint NOT NULL,
    total_geoproduct_button_click bigint NOT NULL
);

CREATE index events_stat_precalced_biz_id_btree_idx
    ON events_stat_precalced (biz_id);

CREATE index events_stat_precalced_last_review_rating_btree_idx
    ON events_stat_precalced (biz_id, last_review_rating);

CREATE index events_stat_precalced_lead_id_btree_idx
    ON events_stat_precalced (lead_id);

CREATE index events_stat_precalced_recent_total_count_btree_idx
    ON events_stat_precalced (biz_id, recent_total_count);

-- other

CREATE FUNCTION last_not_null_transition_func(state text, val text)
RETURNS text LANGUAGE SQL AS $$
    SELECT COALESCE(val, state)
$$;

CREATE FUNCTION last_not_null_final_func(state text)
RETURNS text LANGUAGE SQL AS $$
    select state;
$$;

CREATE AGGREGATE last_not_null_value(text) (
    sfunc = last_not_null_transition_func,
    stype = text,
    finalfunc = last_not_null_final_func
);
