from typing import List

from maps_adv.geosmb.tuner.server.lib.data_manager import BaseDataManager
from maps_adv.geosmb.tuner.server.lib.exceptions import UnknownBizId
from maps_adv.geosmb.tuner.server.lib.enums import PermissionFlag


class Domain:
    _dm: BaseDataManager

    def __init__(self, dm: BaseDataManager):
        self._dm = dm

    async def fetch_settings(self, biz_id: int) -> dict:
        try:
            return await self._dm.fetch_settings(biz_id=biz_id)
        except UnknownBizId:
            return await self._create_default_settings(biz_id=biz_id)

    async def _create_default_settings(self, biz_id: int) -> dict:
        # TODO здесь мы будем добывать первоначальный email
        #  и создавать настройки с ним

        return await self._dm.create_default_settings(
            biz_id=biz_id, emails=None, phone=None
        )

    async def update_settings(self, biz_id: int, **kwargs) -> dict:
        if not await self._dm.check_settings_exist(biz_id=biz_id):
            raise UnknownBizId(f"Settings for biz_id {biz_id} not found.")

        return await self._dm.update_settings(biz_id=biz_id, **kwargs)

    async def fetch_settings_v2(self, biz_id: int) -> dict:
        try:
            return await self._dm.fetch_settings_v2(biz_id=biz_id)
        except UnknownBizId:
            return await self._create_default_settings_v2(biz_id=biz_id)

    async def _create_default_settings_v2(self, biz_id: int) -> dict:
        # TODO здесь мы будем добывать первоначальный email
        #  и создавать настройки с ним

        return await self._dm.create_default_settings_v2(
            biz_id=biz_id, contacts={}
        )

    async def update_settings_v2(self, biz_id: int, **kwargs) -> dict:
        if not await self._dm.check_settings_exist(biz_id=biz_id):
            raise UnknownBizId(f"Settings for biz_id {biz_id} not found.")

        return await self._dm.update_settings_v2(biz_id=biz_id, **kwargs)

    async def fetch_permissions(self, biz_id: int) -> dict:
        permissions = await self._dm.fetch_permissions(biz_id=biz_id)
        return {'permissions': permissions or []}

    async def check_permission(self, biz_id: int, passport_uid: int, flag: PermissionFlag) -> dict:
        has_permission = await self._dm.check_permission(
            biz_id=biz_id,
            passport_uid=passport_uid,
            flag=flag,
        )
        return dict(has_permission=has_permission)

    async def update_permission(self, biz_id: int, passport_uid: int, flags: List[PermissionFlag]) -> dict:
        return await self._dm.update_permission(
            biz_id=biz_id,
            passport_uid=passport_uid,
            flags=flags,
        )

    async def fetch_telegram_users(self, user_logins: List[str]) -> List[dict]:
        users = await self._dm.fetch_telegram_users(user_logins=user_logins)
        return {'users': users or []}

    async def check_telegram_user(self, user_login: str) -> dict:
        is_authorized = await self._dm.check_telegram_user(user_login=user_login)
        return dict(is_authorized=is_authorized)

    async def update_telegram_user(self, user_id: int, user_login: str) -> dict:
        return await self._dm.update_telegram_user(
            user_id=user_id,
            user_login=user_login,
        )

    async def delete_telegram_user(self, user_id: int) -> None:
        await self._dm.delete_telegram_user(
            user_id=user_id,
        )
