from alembic.operations import MigrateOperation, Operations


class ReplacableObject:
    def __init__(self, name, sqltext):
        self.name = name
        self.sqltext = sqltext


class ReversibleOp(MigrateOperation):
    def __init__(self, target):
        self.target = target

    @classmethod
    def invoke_for_target(cls, operations, target):
        op = cls(target)
        return operations.invoke(op)

    def reverse(self):
        raise NotImplementedError()

    @classmethod
    def _get_object_from_version(cls, operations, ident):
        version, objname = ident.split(".")

        module = operations.get_context().script.get_revision(version).module
        obj = getattr(module, objname)
        return obj

    @classmethod
    def replace(cls, operations, target, replaces=None, replace_with=None):

        if replaces:
            old_obj = cls._get_object_from_version(operations, replaces)
            drop_old = cls(old_obj).reverse()
            create_new = cls(target)
        elif replace_with:
            old_obj = cls._get_object_from_version(operations, replace_with)
            drop_old = cls(target).reverse()
            create_new = cls(old_obj)
        else:
            raise TypeError("replaces or replace_with is required")

        operations.invoke(drop_old)
        operations.invoke(create_new)


@Operations.register_operation("create_materialized_view", "invoke_for_target")
@Operations.register_operation("replace_materialized_view", "replace")
class CreateMaterializedViewOp(ReversibleOp):
    def reverse(self):
        return DropMaterializedViewOp(self.target)


@Operations.register_operation("drop_materialized_view", "invoke_for_target")
class DropMaterializedViewOp(ReversibleOp):
    def reverse(self):
        return CreateMaterializedViewOp(self.target)


@Operations.implementation_for(CreateMaterializedViewOp)
def create_materialized_view(operations, operation):
    name = operation.target.name
    sql = operation.target.sqltext

    operations.execute(f"CREATE MATERIALIZED VIEW {name} AS {sql}")


@Operations.implementation_for(DropMaterializedViewOp)
def drop_materialized_view(operations, operation):
    name = operation.target.name

    operations.execute(f"DROP MATERIALIZED VIEW {name}")
