import pytest
from marshmallow import ValidationError

from maps_adv.stat_controller.server.lib.api_providers import Charger
from maps_adv.stat_controller.server.tests.tools import coro_mock, dt

pytestmark = [pytest.mark.asyncio]


@pytest.fixture
def domain():
    class Domain:
        update = coro_mock()

    return Domain()


@pytest.fixture
def provider(domain):
    return Charger(domain)


@pytest.mark.parametrize(
    "update",
    (
        {"task_id": ""},
        {"task_id": None},
        {"status": ""},
        {"status": None},
        {"status": "some_status"},
        {"executor_id": ""},
        {"executor_id": None},
    ),
)
async def test_raises_for_invalid_input(update, provider):
    data = {"task_id": 10, "status": "data_received", "executor_id": 20}
    data.update(update)

    with pytest.raises(ValidationError):
        await provider.update(**data)


async def test_returns_task_data(domain, provider):
    domain.update.coro.return_value = {
        "id": 1,
        "timing_from": dt("1999-05-05 04:40:00"),
        "timing_to": dt("1999-05-05 04:59:31"),
    }

    got = await provider.update(
        executor_id="executor0",
        task_id=1,
        status="context_received",
        execution_state="some_state",
    )

    assert got == {
        "id": 1,
        "timing_from": "1999-05-05T04:40:00+00:00",
        "timing_to": "1999-05-05T04:59:31+00:00",
    }
