from datetime import timedelta
from typing import List, Optional

from maps_adv.adv_store.client import Client as AdvStoreClient
from maps_adv.billing_proxy.client import Client as BillingProxyClient
from maps_adv.statistics.beekeeper.lib.packet_size_calculator import (
    PacketSizeCalculator,
)
from maps_adv.statistics.beekeeper.lib.steps import (
    AdvStoreNotification,
    BillingNotification,
    ChargesCalculator,
    ContextCollector,
    FreeEventsPorter,
    MarkProcessedPacket,
    PaidEventsPorter,
)
from maps_adv.warden.client.lib import Pipeline, Step


def create_charger(
    *,
    lag_packet_size: timedelta,
    min_packet_size: timedelta,
    max_packet_size: timedelta,
    time_threshold_free_events: timedelta,
    ch_config: dict,
    adv_store_client: AdvStoreClient,
    billing_proxy_client: BillingProxyClient,
    campaigns_for_processing: Optional[List[int]] = None,  # GEOPROD-4108
    build_revision: int,
    ch_max_memory_usage: int,
    ch_query_id: str = "",
) -> Pipeline:
    return Pipeline(
        [
            Step(
                "context",
                ContextCollector(
                    packet_size_calculator=PacketSizeCalculator(
                        time_lag=lag_packet_size,
                        min_packet_size=min_packet_size,
                        max_packet_size=max_packet_size,
                        **ch_config,
                    ),
                    adv_store_client=adv_store_client,
                    billing_proxy_client=billing_proxy_client,
                    ch_client_params=ch_config,
                    ch_query_id=ch_query_id,
                    campaigns_for_processing=campaigns_for_processing,  # GEOPROD-4108
                ),
            ),
            Step("calculator", ChargesCalculator()),
            Step("billing", BillingNotification(billing_proxy_client)),
            Step(
                "paid_events_porter",
                PaidEventsPorter(
                    ch_client_params=ch_config,
                    ch_max_memory_usage=ch_max_memory_usage,
                    ch_query_id=ch_query_id,
                ),
            ),
            Step("adv_store", AdvStoreNotification(adv_store_client)),
            Step(
                "free_events_porter",
                FreeEventsPorter(
                    ch_client_params=ch_config,
                    ch_max_memory_usage=ch_max_memory_usage,
                    event_group_id_time_threshold=time_threshold_free_events,
                    ch_query_id=ch_query_id,
                ),
            ),
            Step(
                "mark_processed_packet",
                MarkProcessedPacket(
                    ch_client_params=ch_config,
                    build_revision=build_revision,
                    ch_query_id=ch_query_id,
                ),
            ),
        ]
    )
