#!/usr/bin/env python3

"""
How to install:
Add symlink to script from arcadia to bin directory on your machine
$ sudo ln -s $(arc root)/partner/bin/sbgrep.py /usr/local/bin/sbgrep

Examples:

Find 5xx causes in java app logs:
$ sbgrep <id> jsonapi exception -i

Grep for testapi truncate timings:
$ sbgrep <id> testapi truncate -i

Show only debug info:
$ sbgrep <id> <app> <pattern> -v 1>/dev/null

"""

import requests
import argparse
import re
import asyncio
import concurrent
import logging
from datetime import datetime
import sys

logging.basicConfig(
    stream=sys.stderr,
    level=logging.INFO,
    format='%(levelname)s:%(message)s'
)
logger = logging.getLogger(__name__)

HOST = 'https://proxy.sandbox.yandex-team.ru/'
INSTANCES = 7

file_names = {
    'jsonapi' : 'docker-backend_jsonapi_java-%s.log',
    'testapi' : 'docker-backend_testapi_java-%s.log',
    'mysql' : 'pi-mysql-%s/mysql/slow-query.log',
    'mock' : 'docker-tests-%s.log',
    'node' : 'docker-frontend_node-%s.log'
}

parser = argparse.ArgumentParser(
    description=__doc__,
    formatter_class=argparse.RawTextHelpFormatter
)

def resource_id(v):
    if requests.get(HOST + v).status_code != 200:
        raise ValueError
    return v

parser.add_argument('resource_id', type=resource_id, help="Log resource id")
parser.add_argument('app', choices=file_names.keys())
parser.add_argument('pattern')
parser.add_argument('-i', action='store_true', help="ignore-case mode")
parser.add_argument('-v', '--verbose', action='store_true')
# TODO: implement --context argument
# man grep: "Print num lines of leading and trailing context surrounding each match."

def get_log(app, idx):
    url = HOST + args.resource_id + '/' + file_names[app] % idx
    logger.debug(f"Start getting {app} logs. Url is {url}")

    start = datetime.now()
    resp = requests.get(url)
    end = datetime.now()
    logger.debug(f"Got log file for {app} in {(end - start).total_seconds() * 1000:0.2f}ms. Instance id {idx}")
    return resp.text

async def main(args):
    executor = concurrent.futures.ThreadPoolExecutor(max_workers=INSTANCES)
    loop = asyncio.get_event_loop()
    logs = await asyncio.gather(
        *[loop.run_in_executor(executor, get_log, args.app, i) for i in range(1, INSTANCES+1)]
    )
    pattern = re.compile(args.pattern, re.IGNORECASE) if args.i else re.compile(args.pattern)

    start = datetime.now()
    for idx, log in enumerate(logs, 1):
        for line in log.split('\n'):
            if pattern.findall(line):
                line = re.sub(pattern, lambda m: '\x1B[31m{}\x1B[0m'.format(m.group()), line)
                print(f'{file_names[args.app] % idx}:{line}')
    end = datetime.now()
    logger.debug(f"Parsing took {(end - start).total_seconds() * 1000:0.2f}ms")

if __name__ == '__main__':
    args = parser.parse_args()
    if args.verbose:
        logger.setLevel(logging.DEBUG)

    asyncio.run(main(args))

