import {includes, trim} from 'lodash';
import {push} from 'react-router-redux';
import {setError, resetError} from './errorActions';
import {fetchBetaList} from './indexActions';
import {fetchJSON, memoizedFetchJSON, simpleFetch} from '../lib/http';

export const CREATE_FORM_FIELD_CHANGE = 'CREATE_FORM_FIELD_CHANGE';

const INFINITY_CHAR = '\u221E';
const SECONDS_IN_HOUR = 60 * 60;

export function createFormFieldChange(fieldName, fieldValue) {
    return dispatch => {
        dispatch({
            type: CREATE_FORM_FIELD_CHANGE,
            fieldName,
            fieldValue
        });
    };
}

export const REQUEST_CREATE_BETA = 'REQUEST_CREATE_BETA';

function _requestCreateBeta() {
    return {
        type: REQUEST_CREATE_BETA
    };
}

export const BETA_CREATED = 'BETA_CREATED';

function _betaCreated(dc, port) {
    return dispatch => {
        dispatch(fetchBetaList(dc));
        dispatch(push(`/beta/${dc}/${port}`));
        dispatch({
            type: BETA_CREATED,
            dc,
            port
        });
    };
}

export const END_CREATE_BETA = 'END_CREATE_BETA';

function _endCreateBeta(dc) {
    return {
        type: END_CREATE_BETA,
        dc
    };
}

export function createBetaRequest(dc, port, body) {
    return async (dispatch) => {

        try {
            const response = await simpleFetch(`/api/${dc}/3/betas/${port}`, {
                credentials: 'same-origin',
                method: 'PUT',
                headers: {
                    'Accept': 'application/json',
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(body)
            });
            const data =  await response.json();
            if (data.status !== 'ok') {
                throw new Error(data);
            }
            dispatch(_betaCreated(dc, port));
        } catch(error) {
            dispatch(setError(error));
        } finally {
            dispatch(_endCreateBeta(dc));
        }
    }
}

export const CREATE_BETA = 'CREATE_BETA';

export function createBeta() {
    return (dispatch, getState) => {
        const {createForm} = getState();
        const {port, dc} = createForm;

        const errors = _validateCreateForm(getState);
        if (errors !== null) {
            return dispatch(_showValidationErrors(errors));
        }

        dispatch(resetError());
        dispatch(_requestCreateBeta());

        const db = {};
        if (createForm.db === 'docker') {
            Object.assign(db, {
                type: 'docker',
                docker_image: createForm.dbImage,
                migrationsql_files: createForm.selectedMigrationSqlFiles,
                migrationscript_files: createForm.selectedMigrationScriptFiles,
                with_manual: Boolean(createForm.withManual),
            });
        } else {
            Object.assign(db, {
                type: 'preset',
                preset_id: createForm.db
            });
        }

        const body = {
            backend: createForm.backendBranch,
            frontend: createForm.frontendBranch,
            java: createForm.javaBranch,
            db,
            blackbox: {type: 'url', url: createForm.blackbox},
            yacotools: {type: 'url', url: createForm.yacotools},
            bs: {type: 'url', url: createForm.bs},
            adfox_host: {type: 'url', url: createForm.adfox_host},
            comment: createForm.comment,
            ubuntu_version: createForm.ubuntuVersion
        };

        if (createForm.ttl !== INFINITY_CHAR) {
            Object.assign(body, {ttl: Number(createForm.ttl) * SECONDS_IN_HOUR});
        }

        dispatch(createBetaRequest(dc, port, body));
    };
}

export const REQUEST_MIGRATION_FILES = 'REQUEST_MIGRATION_FILES';

export const RECEIVE_MIGRATION_FILES = 'RECEIVE_MIGRATION_FILES';

export function _receiveMigrationFiles(dc, branch, migrationSqlFiles, migrationScriptFiles) {
    return {
        type: RECEIVE_MIGRATION_FILES,
        branch,
        dc,
        migrationSqlFiles,
        migrationScriptFiles,
    };
}

export const FETCH_DB_IMAGES = 'FETCH_DB_IMAGES';

export function fetchDbImages(dc) {
    return dispatch => {
        dispatch(_requestDbImages(dc));

        fetchJSON(`/api/${dc}/3/cached_docker_db_images`)
            .then((dbImages) => dispatch(_receiveDbImages(dc, dbImages)));
    };
}

export const REQUEST_DB_IMAGES = 'REQUEST_DB_IMAGES';

function _requestDbImages(dc) {
    return {
        type: 'REQUEST_DB_IMAGES',
        dc
    };
}

export const RECEIVE_DB_IMAGES = 'RECEIVE_DB_IMAGES';

export function _receiveDbImages(dc, dbImages) {
    return {
        type: RECEIVE_DB_IMAGES,
        dc,
        dbImages,
    };
}

export const FETCH_PRODUCTION_TAGS = 'FETCH_PRODUCTION_TAGS';

export function fetchProductionTags() {
    return dispatch => {
        dispatch(_requestProductionTags());

        memoizedFetchJSON(`/partner2_production_version.json`)
            .then((result) => dispatch(_receiveProductionTags(result.s, result.f)));
    };
}

export const REQUEST_PRODUCTION_TAGS = 'REQUEST_PRODUCTION_TAGS';

function _requestProductionTags() {
    return {
        type: REQUEST_PRODUCTION_TAGS
    };
}

export const RECEIVE_PRODUCTION_TAGS = 'RECEIVE_PRODUCTION_TAGS';

export function _receiveProductionTags(backendTag, frontendTag) {
    return {
        type: RECEIVE_PRODUCTION_TAGS,
        backendTag,
        frontendTag
    };
}

function _validateCreateForm(getState) {
    const errors = {};
    const {createForm} = getState();
    const {
        backendBranch,
        db,
        dbImage,
        frontendBranch,
        javaBranch,
        port,
        ttl,
    } = createForm;

    if (port === null) {
        errors.port = 'Выберите порт';
    }

    if (db === 'docker' && dbImage === null) {
        errors.dbImage = 'Выберите образ БД';
    }

    if (!_isBranchValid(frontendBranch)) {
        errors.frontendBranch = 'Выберите ветку фронтенда';
    }

    if (!_isBranchValid(backendBranch)) {
        errors.backendBranch = 'Выберите ветку бэкенда';
    }

    if (!_isBranchValid(javaBranch)) {
        errors.javaBranch = 'Некорректно указана Java-ветка';
    }

    if (!_isTtlValid(ttl)) {
        errors.ttl = `Время жизни должно быть натуральным числом, но не более чем 365 (или ${INFINITY_CHAR})`;
    }

    if (Object.keys(errors).length === 0) {
        return null;
    }

    return errors;
}

function _isTtlValid(ttl) {
    if (ttl === INFINITY_CHAR) {
        return true;
    }

    const ttlNumber = Number(ttl);

    return trim(ttl) !== ''
        && !isNaN(ttlNumber)
        && Math.floor(ttlNumber) === ttlNumber
        && ttlNumber > 0
        && ttlNumber <= 365 * 24;
}

function _isBranchValid(branch) {
    if (!branch || branch === 'trunk') {
        return true;
    }

    const userPattern = /^users\/[\w-]+\/[\w.-]+$/;
    const releasePattern = /^releases\/partner\/[\w.-]+$/;
    const tagsPattern = /^tags\/releases\/partner\/(yharnam\/)?[\w.-]+$/;
    const s = branch.trim();
    return userPattern.test(s) || releasePattern.test(s) || tagsPattern.test(s);
}

export const SHOW_VALIDATION_ERRORS = 'SHOW_VALIDATION_ERRORS';

function _showValidationErrors(errors) {
    return {
        type: SHOW_VALIDATION_ERRORS,
        errors
    };
}
