const FETCH_LOG_INTERVAL = 5000;

let logIntervalId = null;

import {setError} from './errorActions';
import {fetchBetaList, updateBetaInIndex} from './indexActions';
import {createBetaRequest} from './createBetaFormActions';
import {fetchJSON, simpleFetch} from '../lib/http';

export const SELECT_BETA = 'SELECT_BETA';

export function selectBeta(dc, port) {
    return dispatch => {
        dispatch({
            type: SELECT_BETA,
            dc,
            port
        });

        dispatch(_clearFetchTimeout());

        dispatch(_fetchLog(dc, port));
        logIntervalId = setTimeout(function onTimeout() {
            dispatch(_fetchLog(dc, port)).then(() => {
                logIntervalId = setTimeout(onTimeout, FETCH_LOG_INTERVAL);
            });
        }, FETCH_LOG_INTERVAL);
    }
}

export const UNSELECT_BETA = 'UNSELECT_BETA';

export function unselectBeta() {
    return dispatch => {
        dispatch({
            type: UNSELECT_BETA
        });
        dispatch(_clearFetchTimeout());
    };
}


const CLEAR_FETCH_TIMEOUT = 'CLEAR_FETCH_TIMEOUT';

function _clearFetchTimeout() {
    return dispatch => {
        dispatch({type: CLEAR_FETCH_TIMEOUT});

        if (logIntervalId) {
            clearInterval(logIntervalId);
        }
    };
}

function _fetchLog(dc, port) {
    return dispatch => {
        return Promise.all([
            simpleFetch(`/api/${dc}/3/betas/log/${port}?tail=40`, {credentials: 'same-origin'})
                .then(response => response.text()),
            fetchJSON(`/api/${dc}/3/betas`)
                .then(betas => betas.filter(beta => Number(beta.port) === Number(port))[0])
        ])
            .then(([log, beta]) => {
                dispatch(_receiveLog(dc, port, log));
                dispatch(updateBetaInIndex(dc, beta));

                if (beta.status !== 'building') {
                    dispatch(_clearFetchTimeout());
                }
            })

    };
}

export const RECEVE_LOG = 'RECEVE_LOG';

function _receiveLog(dc, port, log) {
    return {
        type: RECEVE_LOG,
        dc,
        port,
        log
    };
}

export function deleteBeta() {
    return async (dispatch, getState) => {
        const {dc, port} = getState().details;

        try {
            dispatch(_requestDeleteBeta(dc, port));
            const data = await fetchJSON(`/api/${dc}/3/betas/${port}`, {method: 'DELETE'});
            if (data.status !== 'ok') {
                throw new Error(data);
            }
            dispatch(_betaDeleted(dc, port));
        } catch(error) {
            dispatch(setError(error));
        } finally {
            dispatch(_endDeleteBeta(dc));
        }
    };
}

export function updateBeta(body) {
    return async (dispatch, getState) => {
        if (body.ttl === null) {
            delete body.ttl;
        }
        const {dc, port} = getState().details;

        await dispatch(deleteBeta())
        await dispatch(createBetaRequest(dc, port, body))
    }
}

export const REQUEST_DELETE_BETA = 'REQUEST_DELETE_BETA';

function _requestDeleteBeta(dc, port) {
    return {
        type: REQUEST_DELETE_BETA,
        dc,
        port
    };
}

export const BETA_DELETED = 'BETA_DELETED';

function _betaDeleted(dc, port) {
    return dispatch => {
        dispatch({
            type: BETA_DELETED,
            dc,
            port
        });

        dispatch(unselectBeta());
        dispatch(fetchBetaList(dc));
    }
}

export const END_DELETE_BETA = 'END_DELETE_BETA';

function _endDeleteBeta(dc) {
    return {
        type: END_DELETE_BETA,
        dc
    };
}

export function makeBetaAction(action, dc, port) {
    return (dispatch, getState) => {
        dispatch(_requestBetaAction(dc, port));

        fetchJSON(`/api/${dc}/3/betas/${port}/action/${action}`, {method: 'PUT'})
            .then(response => {
                if (response.status !== 'ok') {
                    return Promise.reject(response);
                }

                dispatch(_betaActionMade(dc, port));
            })
            .catch(error => dispatch(setError(error)))
            .then(() => dispatch(_endMakeAction(dc)));
    };
}

export const REQUEST_BETA_ACTION = 'REQUEST_BETA_ACTION';
function _requestBetaAction(dc, port) {
    return {type: REQUEST_BETA_ACTION, dc, port};
}

export const BETA_ACTION_MADE = 'BETA_ACTION_MADE';
function _betaActionMade(dc, port) {
    return dispatch => {
        dispatch({type: BETA_ACTION_MADE, dc, port});

        dispatch(fetchBetaList(dc));
    };
}

export const END_MAKE_ACTION = 'END_MAKE_ACTION';
function _endMakeAction(dc) {
    return {type: END_MAKE_ACTION, dc};
}
