import React, {PureComponent} from 'react';

import moment from 'moment';
import 'moment/locale/ru';
import reactAnsiStyle from 'react-ansi-style';
import {isEqual, pick} from 'lodash';
import {parse as urlParse} from 'url';

import Link from 'teatime-components/component/Link';
import Button from 'teatime-components/component/Button';

import BetaDBConfig from '../BetaDBConfig/BetaDBConfig';
import BetaDBConnectCommand from '../BetaDBConnectCommand/BetaDBConnectCommand';
import TimeToLiveLabel from '../TimeToLiveLabel/TimeToLiveLabel';
import BetaStatus from '../BetaStatus/BetaStatus';

import 'react-ansi-style/inject-css';
import {DASH, INFINITY_CHAR} from '../symbols.js';

import './BetaDetails.css';
import {BetaType} from '../../types';

const SECONDS_IN_HOUR = 60 * 60;

const REPOS_COMMIT_LINK = {
    frontend: 'https://arcanum.yandex-team.ru/arc_vcs/commit/',
    backend: 'https://arcanum.yandex-team.ru/arc_vcs/commit/',
    java: 'https://arcanum.yandex-team.ru/arc_vcs/commit/',
};
const TICKET_REGEXP = /\b((?:PI|INFRASTRUCTUREPI)-\d+).*$/;

const betaSettingsKeys = [
    'adfox_host',
    'ubuntu_version',
    'backend',
    'blackbox',
    'bs',
    'comment',
    'db',
    'frontend',
    'ttl',
    'yacotools',
    'java',
];

type BetaDetailsProps = {
    beta: BetaType;
    log: string;
    dc: string;
    port: string;

    pendingPort: unknown;
    pendingDc: unknown;
    listIsPending: boolean;

    productionBackendTag: string;
    productionFrontendTag: string;

    selectedBeta: BetaType;

    onBetaSelect: (beta: BetaType) => void;
    onBetaUnselect: () => void;

    onDeleteClick: () => void;
    onStartClick: () => void;
    onStopClick: () => void;
    onUpdateClick: (body: Partial<BetaType>) => void;
};

class BetaDetails extends PureComponent<BetaDetailsProps> {
    componentDidMount() {
        const {selectedBeta, onBetaSelect} = this.props;
        if (onBetaSelect) {
            onBetaSelect(selectedBeta);
        }
    }

    componentWillUnmount() {
        const {onBetaUnselect} = this.props;
        if (onBetaUnselect) {
            onBetaUnselect();
        }
    }

    componentWillReceiveProps(nextProps) {
        if (!isEqual(this.props.selectedBeta, nextProps.selectedBeta)) {
            const {onBetaSelect} = this.props;

            if (onBetaSelect) {
                onBetaSelect(nextProps.selectedBeta);
            }
        }
    }

    onUpdateClick = () => {
        const {beta, onUpdateClick} = this.props;
        
        const body = pick(beta, betaSettingsKeys);

        onUpdateClick(body);
    }

    render() {
        const {
            dc,

            beta,

            pendingPort,
            pendingDc,
            listIsPending,

            onDeleteClick,
            onStartClick,
            onStopClick,
        } = this.props;

        if (listIsPending) {
            return null;
        } else if (!beta) {
            return this._render404Error();
        } else if (beta.status === 'occupied') {
            return this._renderOccupiedError();
        }

        const {comment, status, port} = beta;
        const header = comment
            ? `${comment} (${port}:${dc})`
            : `Бета ${port}:${dc}`;

        const isPending =
            Number(port) === Number(pendingPort) && dc === pendingDc;

        return (
            <div className='BetaDetails'>
                <h2>
                    {header}&nbsp;
                    <BetaStatus status={status} />
                </h2>

                <div className='BetaDetails__buttons'>
                    <Button
                        className='BetaDetails__button'
                        disabled={status === 'running' || isPending}
                        onClick={onStartClick}
                    >
                        Старт
                    </Button>
                    <Button
                        className='BetaDetails__button'
                        disabled={status === 'stopped' || isPending}
                        onClick={onStopClick}
                    >
                        Стоп
                    </Button>
                    <Button
                        className='BetaDetails__delete-button BetaDetails__button'
                        disabled={isPending}
                        onClick={onDeleteClick}
                    >
                        Удалить
                    </Button>
                    <Button
                        className='BetaDetails__button'
                        disabled={isPending}
                        onClick={this.onUpdateClick}
                    >
                        Пересобрать
                    </Button>
                </div>

                <div className='BetaDetails__columns'>
                    {this._renderLeftColumn()}
                    {this._renderRightColumn()}
                </div>
            </div>
        );
    }

    _renderLeftColumn() {
        const {
            beta: {
                backend,
                blackbox,
                bs,
                created,
                created_by: createdBy,
                db,
                frontend,
                java,
                port,
                ttl,
                url,
                yacotools,
                rosetta_debugger_url,
                ubuntu_version,
                comment,
            },
            dc,
            productionBackendTag,
            productionFrontendTag,
        } = this.props;

        const backendCommitLink = this._getCommitLink('backend');
        const frontendCommitLink = this._getCommitLink('frontend');
        const javaCommitLink = this._getCommitLink('java');
        const backendTicketLink = this._getTicketLinkFromBranch(backend);
        const javaTicketLink = this._getTicketLinkFromBranch(java);
        const frontendTicketLink = this._getTicketLinkFromBranch(frontend);
        const {hostname} = urlParse(url);

        let ttlLabel = null;
        if (ttl) {
            ttlLabel = (
                <span>
                    {Math.floor(ttl / SECONDS_IN_HOUR)}ч. (
                    <TimeToLiveLabel created={created} ttl={ttl} />)
                </span>
            );
        } else {
            ttlLabel = INFINITY_CHAR;
        }

        return (
            <div className='BetaDetails__left-column'>
                <div className='BetaDetails__header'>О бете</div>

                <BetaDetailsRow label='Ссылка'>
                    <Link target='_blank' href={url}>
                        {url}
                    </Link>
                </BetaDetailsRow>

                <BetaDetailsRow label='Владелец'>
                    {createdBy || '—'}
                </BetaDetailsRow>

                <BetaDetailsRow label='Описание'>
                    {comment || '—'}
                </BetaDetailsRow>

                <BetaDetailsRow label='Время создания'>
                    {created && moment(created).format('D MMMM YYYY HH:mm')} (
                    {created && moment(created).fromNow()})
                </BetaDetailsRow>

                <BetaDetailsRow label='Время жизни'>{ttlLabel}</BetaDetailsRow>

                <BetaDetailsRow label='Путь на машине'>
                    {`/local/creator/${port}/`}
                </BetaDetailsRow>

                <div className='BetaDetails__header'>Ветки и тикеты</div>

                <BetaDetailsRow label='Версия Ubuntu'>
                    {ubuntu_version}
                </BetaDetailsRow>

                <BetaDetailsRow label='Бэкенд Perl'>
                    {this._getBranchName(backend, productionBackendTag)}{' '}
                    {backendCommitLink}
                </BetaDetailsRow>

                <BetaDetailsRow label='Бэкенд Java'>
                    {this._getBranchName(java, null)} {javaCommitLink}
                </BetaDetailsRow>

                <BetaDetailsRow label='Тикет бэкенда'>
                    {backendTicketLink || javaTicketLink
                        ? [backendTicketLink, javaTicketLink]
                        : DASH}
                </BetaDetailsRow>

                <BetaDetailsRow label='Фронтенд'>
                    {this._getBranchName(frontend, productionFrontendTag)}{' '}
                    {frontendCommitLink}
                </BetaDetailsRow>

                <BetaDetailsRow label='Тикет фронтенда'>
                    {frontendTicketLink || DASH}
                </BetaDetailsRow>

                <div className='BetaDetails__header'>База данных</div>

                <BetaDetailsRow label='Тип'>{db && db.type}</BetaDetailsRow>

                {db && db.docker_image ? (
                    <BetaDetailsRow label='Образ'>
                        {db.docker_image}
                    </BetaDetailsRow>
                ) : null}

                {db && db.migrationsql_files ? (
                    <BetaDetailsRow label='SQL-файлы миграции'>
                        {this._getMigrationSqlFiles(db.migrationsql_files)}
                    </BetaDetailsRow>
                ) : null}

                {db && db.migrationscript_files ? (
                    <BetaDetailsRow label='Скрипты миграции'>
                        {this._getMigrationSqlFiles(db.migrationscript_files)}
                    </BetaDetailsRow>
                ) : null}

                {db && db.preset_id ? (
                    <BetaDetailsRow label='Пресет'>
                        {db.preset_id}
                    </BetaDetailsRow>
                ) : null}

                {db && db.type === 'docker' && (
                    <div>
                        <div className='BetaDetails__header'>Конфиг БД</div>
                        <BetaDBConfig port={port} dc={dc} />
                        <div className='BetaDetails__header'>
                            Команда подключения к БД
                        </div>
                        <BetaDBConnectCommand port={port} dc={dc} />
                    </div>
                )}

                <div className='BetaDetails__header'>Сторонние сервисы</div>

                <BetaDetailsRow label='Yacotools'>
                    {yacotools && yacotools.url}
                </BetaDetailsRow>

                <BetaDetailsRow label='БК'>{bs && bs.url}</BetaDetailsRow>

                <BetaDetailsRow label='Blackbox'>
                    {blackbox && blackbox.url}
                </BetaDetailsRow>

                {/* TODO адфокс не приходит с бека */}
                {/* <BetaDetailsRow label='Adfox'></BetaDetailsRow> */}
            </div>
        );
    }

    _renderRightColumn() {
        const {beta, dc, log} = this.props;
        const {port} = beta;

        return (
            <div className='BetaDetails__right-column'>
                <div className='BetaDetails__header'>
                    Лог ({this._getLogLink(dc, port, 'полный')})
                </div>
                <pre className='BetaDetails__log'>
                    {log && reactAnsiStyle(React, log)}
                </pre>
            </div>
        );
    }

    _getBranchName(branch, productionTag) {
        if (!branch) {
            return DASH;
        }
        if (branch === productionTag) {
            return `${branch} (текущий продакшн)`;
        }

        return branch;
    }

    _getCommitLink(repoName) {
        const commit = this.props.beta[`${repoName}_sha1`];
        if (!commit) {
            return null;
        }

        const repoLink = `${REPOS_COMMIT_LINK[repoName]}${commit}`;

        return (
            <Link target='_blank' href={repoLink}>
                {commit.substr(0, 5)}
            </Link>
        );
    }

    _getTicketLinkFromBranch(branchName) {
        const groups = TICKET_REGEXP.exec(branchName);

        if (!groups) {
            return null;
        }

        const ticket = groups[1];

        return (
            <Link target='_blank' href={`https://st.yandex-team.ru/${ticket}`}>
                {ticket}
            </Link>
        );
    }

    _render404Error() {
        const link = <Link href='/'>список бет </Link>;

        const {port, dc} = this.props;

        return (
            <div className='BetaDetails__error'>
                <div className='BetaDetails__error-content'>
                    <div className='BetaDetails__img-container'>
                        <i className='BetaDetails__wiedzmin-missing' />
                    </div>
                    <h2>Бета не найдена</h2>
                    Беты {dc}:{port} не существует. Вы можете вернуться на{' '}
                    {link}.
                </div>
            </div>
        );
    }

    _renderOccupiedError() {
        const link = (
            <Link
                target='_blank'
                href='https://wiki.yandex-team.ru/partner/w/partner-creator/#statusybetvkreatore'
            >
                Ссылка на&nbsp;подробности
            </Link>
        );

        const {port, dc} = this.props;

        return (
            <div className='BetaDetails__error'>
                <div className='BetaDetails__error-content'>
                    <div className='BetaDetails__img-container'>
                        <i className='BetaDetails__wiedzmin-eh' />
                    </div>
                    <h2>Вот, незадача!</h2>
                    Бета {dc}:{port} оказалась в&nbsp;статусе «occupied». {link}
                    .
                </div>
            </div>
        );
    }

    _getLogLink(dc, port, label) {
        return (
            <Link
                target='_blank'
                href={`https://creator${dc}.partner.yandex-team.ru/api/3/betas/log/${port}`}
            >
                {label}
            </Link>
        );
    }

    _getMigrationSqlFiles(files) {
        const {beta} = this.props;
        const {backend_sha1} = beta;
        return files.map((filePath, i) => {
            const isLast = i === files.length - 1;
            return (
                <span key={filePath}>
                    <Link
                        target='_blank'
                        href={`https://github.yandex-team.ru/partner/partner2/blob/${backend_sha1}/migrations/${filePath}`}
                    >
                        {filePath}
                    </Link>
                    {isLast ? null : ', '}
                </span>
            );
        });
    }
}

function BetaDetailsRow({
    label,
    children,
}: {
    label: string;
    children: React.ReactNode;
}) {
    return (
        <div className='BetaDetails__row'>
            <div className='BetaDetails__label'>{label + ' '}</div>
            <div className='BetaDetails__value'>{children}</div>
        </div>
    );
}

export default BetaDetails;
