/* global StaffCard */


import React, {Component} from 'react';
import moment from 'moment';
import classnames from 'classnames';
import {Link as RouterLink} from 'react-router';

import Link from 'teatime-components/component/Link';

import BetaStatus from '../BetaStatus/BetaStatus';
import TimeToLiveLabel from '../TimeToLiveLabel/TimeToLiveLabel';
import {BetaType} from '../../types';

import {DASH} from '../symbols.js';

import './BetaListItem.css';

const TICKET_REGEXP = /\b([A-Z]+-\d+)/;

function BetaLink({children, url}) {
    if (!url) {
        return <span>{children}</span>;
    }
    return (
        <Link href={url} key='label'>{children}</Link>
    );
}

class TicketLinkText extends Component<{wrapUrl?: string}> {
    render() {
        const {wrapUrl, children} = this.props;

        if (!children) {
            return null;
        }

        const parts = this._getTicketParts(children);

        if (parts) {
            // Odd parts are tickets, converting them to links
            const partsWithLinks = parts.map((v, i) => (
                <span key={i}>
                    {i % 2 ? (
                        this._getLinkForTicket(v)
                    ) : (
                        <BetaLink url={wrapUrl}>{v}</BetaLink>
                    )}
                </span>
            ));
            return <span>{partsWithLinks}</span>;
        }

        return <BetaLink url={wrapUrl}>{children}</BetaLink>;
    }

    _getTicketParts(text) {
        if (typeof text !== 'string') {
            return null;
        }

        const parts = text.split(TICKET_REGEXP);

        if (!parts || parts.length === 1) {
            return null;
        }

        return parts;
    }

    _getLinkForTicket(ticket) {
        if (typeof ticket !== 'string') {
            return null;
        }

        return (
            <Link key={ticket} target='_blank' href={`https://st.yandex-team.ru/${ticket}`}>
                {ticket}
            </Link>
        );
    }
}

type BetaListItemProps =  BetaType & {
    compact?: boolean;
    dc: string;
    selected?: unknown;
    canDelete?: boolean;
    pending?: boolean;
    onDeleteClick: () => void;
};

class BetaListItem extends Component<BetaListItemProps> {
    render() {
        const {
            backend,
            comment,
            compact,
            created,
            created_by: createdBy,
            dc,
            frontend,
            java,
            port,
            ttl,
            rosetta_debugger_url: rosettaDebuggerUrl,
            url: betaUrl,
            selected,
            status,
            ubuntu_version,
        } = this.props;

        const debuggerPort = Number(port) - 200;

        if (status === 'free') {
            return null;
        }

        const backendBranchClassName = this._getBranchClassName('backend');
        const frontendBranchClassName = this._getBranchClassName('frontend');
        const javaBranchClassName = this._getBranchClassName('java');

        const className = classnames('BetaListItem', {
            'BetaListItem_selected': selected === true,
            'BetaListItem_stopped': status === 'stopped'
        });
        return (
            <tr className={className}>
                <td className='BetaListItem__cell'>
                    <BetaStatus status={status}/>
                </td>
                <td className='BetaListItem__cell'>
                    <Link target='_blank' href={betaUrl}>
                        {port}
                    </Link>
                </td>
                <td className='BetaListItem__cell BetaListItem__cell_owner'>
                    <img
                        className='BetaListItem__user-avatar'
                        alt='Avatar'
                        src={`//center.yandex-team.ru/api/v1/user/${createdBy}/avatar/50.jpg`}
                    />
                    <div className='BetaListItem__user' ref='user'>{createdBy}</div>
                </td>
                <td className='BetaListItem__cell'>
                    <span className='BetaListItem__created'>
                        {created && moment(created).fromNow()}
                    </span>
                    <br/>
                    <span className='BetaListItem__created'>
                        {created && moment(created).format('HH:mm D MMM YYYY')}
                    </span>
                    <br/>
                    <TimeToLiveLabel created={created} ttl={ttl}/>
                </td>
                <td className='BetaListItem__cell'>
                    <span className='BetaListItem__info-link'>
                        <TicketLinkText wrapUrl={`/beta/${dc}/${port}`}>{comment}</TicketLinkText>
                    </span>
                </td>
                {
                    compact
                        ? null
                        : <td className='BetaListItem__cell'>
                            <div>partner2 perl: <span className={backendBranchClassName}><TicketLinkText>{backend}</TicketLinkText></span></div>
                            <div>yharnam: <span className={frontendBranchClassName}><TicketLinkText>{frontend}</TicketLinkText></span></div>
                            <div>java: <span className={javaBranchClassName}><TicketLinkText>{java || DASH}</TicketLinkText></span></div>
                            <div>ubuntu: {ubuntu_version}</div>
                        </td>
                }
                {
                    compact
                        ? null
                        : <td className='BetaListItem__cell'>
                            {this._getDBDescription()}
                        </td>
                }
                <td className='BetaListItem__cell'>
                    {/* @ts-ignore */}
                    <RouterLink className='BetaListItem__info-link' to={`/beta/${dc}/${port}`}>
                        <div className='BetaListItem__info'>i</div>
                    </RouterLink>
                </td>
            </tr>
        );
    }

    componentDidMount() {
        const {created_by: createdBy} = this.props;
        const {user} = this.refs;

        new StaffCard(user, createdBy);
    }

    _getBranchClassName(branchName) {
        const status = this.props[`${branchName}_status`];

        if (status === 'deleted') {
            return 'BetaListItem__deleted-branch';
        } else if (status === 'outdated') {
            return 'BetaListItem__outdated-branch';
        }

        return 'BetaListItem__branch';
    }

    _getDBDescription() {
        const {db = ''} = this.props;

        if (typeof db === 'string') {
            return db;
        } else if (db.type === 'preset') {
            return db.preset_id;
        }

        return db.type;
    }
}

export default BetaListItem;
