import React, {Component} from 'react';

import {
    Button,
    Check,
    Input,
    Link,
    Select,
    Spin,
} from 'teatime-components';

import classnames from 'classnames';

import TooltipWrapper from '../TooltipWrapper/TooltipWrapper';
import {SERVICES_URLS} from '../../reducers/createBetaFormReducer';

import {
    generateFrontendBranchByTag,
    generatePertBranchByTag,
} from '../../utils/tags';

import './CreateBetaForm.css';

const SelectWithTooltip = TooltipWrapper(Select); // eslint-disable-line new-cap
const InputWithTooltip = TooltipWrapper(Input); // eslint-disable-line new-cap

const INFINITY_CHAR = '\u221E';

class CreateBetaForm extends Component {

    render() {
        const {
            freePortsOpts,
            dbs,
            ubuntuVersionsOpts,
            ubuntuVersion,
        } = this.props;
        const requiredProps = [freePortsOpts, dbs];

        if (requiredProps.some(prop => prop.length === 0)) {
            return (
                <div className='CreateBetaForm'>
                    <h3>Добавление беты</h3>
                    {this._renderDcField()}
                </div>
            );
        }

        const dbOptions = dbs.map(_mapOptions);

        const {
            port,
            backendBranch,
            frontendBranch,
            javaBranch,
            db,
            blackbox,
            yacotools,
            bs,
            adfox_host,
            comment,
            ttl,

            pending,
            errors,

            onFieldChange,
            onCreateClick
        } = this.props;

        const {port: portError, ttl: ttlError} = errors;
        // Не даем создать бету при наличии миграций на не-доккерной базе
        const needToSwitchDB = this._needToSwitchDB();

        return (
            <div className='CreateBetaForm'>
                <h3>Добавление беты</h3>

                {this._renderDcField()}

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Порт</label>
                    <div className='CreateBetaForm__control'>
                        <SelectWithTooltip
                            className='CreateBetaForm__select'
                            searchable={true}
                            name='port'
                            options={freePortsOpts}
                            value={String(port)}
                            onChange={(_, {value}) => onFieldChange('port', value)}

                            tooltipType='warning'
                            tooltipMessage={portError}
                            tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Версия Ubuntu</label>
                    <div className='CreateBetaForm__control'>
                        <SelectWithTooltip
                            className='CreateBetaForm__select'
                            searchable={true}
                            name='ubuntuVersion'
                            options={ubuntuVersionsOpts}
                            value={ubuntuVersion}
                            onChange={(_, {value}) => onFieldChange('ubuntuVersion', value)}
                            tooltipType='warning'
                            tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Ветка backend Perl</label>
                    <div className='CreateBetaForm__control'>
                        <InputWithTooltip
                            className='CreateBetaForm__wide-input'
                            value={backendBranch}
                            name='backendBranch'
                            onChange={(_, {value}) => onFieldChange('backendBranch', value)}
                            placeholder='trunk'
                            tooltipType='warning'
                            tooltipMessage={errors.backendBranch}
                            tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                        />
                        {this._renderProductionBackendTagSetButton()}
                    </div>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Ветка frontend</label>
                    <div className='CreateBetaForm__control'>
                        <InputWithTooltip
                            className='CreateBetaForm__wide-input'
                            value={frontendBranch}
                            name='frontendBranch'
                            onChange={(_, {value}) => onFieldChange('frontendBranch', value)}
                            placeholder='trunk'
                            tooltipType='warning'
                            tooltipMessage={errors.frontendBranch}
                            tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                        />
                        {this._renderProductionFrontendTagSetButton()}
                    </div>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Ветка Java</label>
                    <div className='CreateBetaForm__control'>
                        <InputWithTooltip
                            className='CreateBetaForm__wide-input'
                            value={javaBranch}
                            name='javaBranch'
                            onChange={(_, {value}) => onFieldChange('javaBranch', value)}
                            placeholder='trunk'
                            tooltipType='warning'
                            tooltipMessage={errors.javaBranch}
                            tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                        />
                    </div>
                    <div className='CreateBetaForm__fieldHint'>
                        Пример:
                        <br />
                        <code>{'users/{user_name}/{branch_name}'}</code>
                        <br />
                        <code>{'releases/partner/{release_branch}'}</code>
                    </div>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>База данных</label>
                    <div className='CreateBetaForm__control'>
                        <Select
                            className='CreateBetaForm__select'
                            name='db'
                            options={dbOptions}
                            value={db}
                            onChange={(_, {value}) => onFieldChange('db', value)}
                        />
                    </div>
                </div>

                {this._renderDbImagesList()}
                {this._renderApplyManualMigrations()}

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Blackbox</label>
                    <div className='CreateBetaForm__control'>
                        <Input
                            value={blackbox}
                            name='blackbox'
                            onChange={(_, {value}) => onFieldChange('blackbox', value)}
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__description'>
                    <label className='CreateBetaForm__label'></label>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('blackbox', SERVICES_URLS.blackbox.real)}>
                        Дев
                    </Button>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('blackbox', SERVICES_URLS.blackbox.mock)}>
                        Мок
                    </Button>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Yacotools</label>
                    <div className='CreateBetaForm__control'>
                        <Input
                            value={yacotools}
                            name='yacotools'
                            onChange={(_, {value}) => onFieldChange('yacotools', value)}
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__description'>
                    <label className='CreateBetaForm__label'></label>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('yacotools', SERVICES_URLS.yacotools.real)}>
                        Дев
                    </Button>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('yacotools', SERVICES_URLS.yacotools.mock)}>
                        Мок
                    </Button>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>БК</label>
                    <div className='CreateBetaForm__control'>
                        <Input
                            value={bs}
                            name='bs'
                            onChange={(_, {value}) => onFieldChange('bs', value)}
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__description'>
                    <label className='CreateBetaForm__label'></label>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('bs', SERVICES_URLS.bs.real)}>
                        Дев
                    </Button>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('bs', SERVICES_URLS.bs.mock)}>
                        Мок
                    </Button>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Adfox</label>
                    <div className='CreateBetaForm__control'>
                        <Input
                            value={adfox_host}
                            name='adfox_host'
                            onChange={(_, {value}) => onFieldChange('adfox_host', value)}
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__description'>
                    <label className='CreateBetaForm__label'></label>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('adfox_host', SERVICES_URLS.adfox_host.default)}>
                        Default
                    </Button>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Описание</label>
                    <div className='CreateBetaForm__control'>
                        <Input
                            value={comment}
                            name='comment'
                            onChange={(_, {value}) => onFieldChange('comment', value)}
                        />
                    </div>
                </div>

                <div className='CreateBetaForm__description'>
                    <label className='CreateBetaForm__label'></label>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('comment', 'Для ревью')}>
                        Для ревью
                    </Button>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('comment', 'Общая бета')}>
                        Общая бета
                    </Button>
                    <Button
                        theme='link'
                        className='CreateBetaForm__suggest'
                        onClick={() => onFieldChange('comment', 'Не удалять')}>
                        Не удалять
                    </Button>
                </div>

                <div className='CreateBetaForm__field'>
                    <label className='CreateBetaForm__label'>Время жизни (в часах)</label>
                    <div className='CreateBetaForm__control'>
                        <InputWithTooltip
                            className='CreateBetaForm__ttl-input'
                            value={ttl}
                            name='ttl'
                            onChange={(_, {value}) => onFieldChange('ttl', value)}
                            tooltipType='warning'
                            tooltipMessage={ttlError}
                            tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                        />
                        <Link
                            className='CreateBetaForm__suggest'
                            onClick={() => onFieldChange('ttl', '24')}>
                            Одни сутки
                        </Link>
                        <Link
                            className='CreateBetaForm__suggest'
                            onClick={() => onFieldChange('ttl', '48')}>
                            Двое суток
                        </Link>
                        <Link
                            className='CreateBetaForm__suggest'
                            onClick={() => onFieldChange('ttl', INFINITY_CHAR)}>
                            Бесконечно
                        </Link>
                    </div>
                </div>

                <Button
                    disabled={pending === true || needToSwitchDB}
                    className='CreateBetaForm__create-button'
                    theme='action'
                    onClick={onCreateClick}
                >
                    Создать
                </Button>
                {pending === true ? <Spin className='CreateBetaForm__spin' size='xs' /> : null}
            </div>
        );
    }

    _renderProductionBackendTagSetButton() {
        const {productionBackendTag, onFieldChange} = this.props;

        const branch = generatePertBranchByTag(productionBackendTag);

        return this._renderProductionTagSetButton(
            productionBackendTag,
            () => onFieldChange('backendBranch', branch)
        );
    }

    _renderProductionFrontendTagSetButton() {
        const {productionFrontendTag, onFieldChange} = this.props;

        const branch = generateFrontendBranchByTag(productionFrontendTag);

        return this._renderProductionTagSetButton(
            productionFrontendTag,
            () => onFieldChange('frontendBranch', branch)
        );
    }

    _renderProductionTagSetButton(tag, handler) {
        const pending = typeof tag !== 'string';

        return (
            <Link
                className={classnames(
                    'CreateBetaForm__prod-tag-set',
                    pending && 'CreateBetaForm__prod-tag-set_disabled'
                )}
                onClick={pending ? () => {} : handler}
            >
                Production: {tag}
            </Link>
        );
    }

    _renderDbImagesList() {
        const {db, dbImagesOpts, dbImage, errors, onFieldChange} = this.props;
        const {dbImage: dbImageError} = errors;

        if (db !== 'docker') {
            return null;
        }

        return (
            <div className='CreateBetaForm__field'>
                <label className='CreateBetaForm__label'>Образ БД</label>
                <div className='CreateBetaForm__control'>
                    <SelectWithTooltip
                        className='CreateBetaForm__select'
                        name='dbImage'
                        options={dbImagesOpts}
                        searchable
                        searchEngine='includes'
                        value={dbImage}
                        onChange={(_, {value}) => onFieldChange('dbImage', value)}
                        tooltipType='warning'
                        tooltipMessage={dbImageError}
                        tooltipWrapper='CreateBetaForm__tooltip-wrapper'
                    />
                </div>
            </div>
        );
    }

    _needToSwitchDB() {
        const {db, migrationSqlFiles} = this.props;
        return (db !== 'docker' && migrationSqlFiles && migrationSqlFiles.length !== 0);
    }

    _getRepoLinkToFile(branch, file) {
        return (
            <span>
                {file}
                <Link target='_blank' href={`https://github.yandex-team.ru/partner/partner2/tree/${branch}/migrations/${file}`}>
                    <span className='CreateBetaForm__externalLink'></span>
                </Link>
            </span>
        );
    }

    _renderApplyManualMigrations() {
        const {
            db,
            withManual,
            onFieldChange,
        } = this.props;

        if (db !== 'docker') {
            return null;
        }

        return (
            <div className='CreateBetaForm__field'>
                <label className='CreateBetaForm__label'>Выполнить MANUAL миграции (если есть)</label>
                <div className='CreateBetaForm__control'>
                    <Check
                        name='withManual'
                        value={withManual}
                        onChange={(_, {checked}) => onFieldChange('withManual', checked)}
                    />
                </div>
            </div>
        )
    }

    _renderDcField() {
        const {dc, dcs, onFieldChange} = this.props;
        const dcsOptions = dcs.map(_mapOptions);
        return (
            <div className='CreateBetaForm__field'>
                <label className='CreateBetaForm__label'>Дата центр</label>
                <div className='CreateBetaForm__control'>
                    <Select
                        className='CreateBetaForm__select'
                        name='dc'
                        options={dcsOptions}
                        searchable={true}
                        value={dc}
                        onChange={(_, {value}) => onFieldChange('dc', value)}
                    />
                </div>
            </div>
        );
    }
}

function _mapOptions(a) {
    return {label: String(a), value: String(a)};
}

export default CreateBetaForm;
