import React, {Component, ReactNode} from 'react';
import {omit} from 'lodash';
import {Tooltip} from 'teatime-components';
import classnames from 'classnames';

import './TooltipWrapper.css';

type TooltipComponentProps = {
    className: string;
    controlClassName: string;
    tooltipClassName: string;
    tooltipDirection: string;
    tooltipMaxWidth: number;
    tooltipMessage: string;
    tooltipSize: string;
    tooltipType: string;
};

type TooltipWrapperOpts = Partial<{
    direction: string;
    className: string;
    maxWidth: number;
    size: unknown;
    type: string;
}>;

export default function TooltipWrapper(
    Target: (...args: any) => JSX.Element,
    {
        className: defaultClassName,
        direction,
        maxWidth,
        size,
        type = 'warning',
    }: TooltipWrapperOpts = {}
) {
    class Wrapper extends Component<TooltipComponentProps> {
        render() {
            const {
                className,
                controlClassName,
                tooltipClassName,
                tooltipDirection,
                tooltipMaxWidth,
                tooltipMessage,
                tooltipSize,
                tooltipType,
            } = this.props;

            const rootClassName = classnames(
                'TooltipWrapper__wrapper',
                defaultClassName,
                className
            );

            const targetClassName = classnames(
                'TooltipWrapper__control',
                className,
                controlClassName
            );

            return (
                <div className={rootClassName}>
                    <Target
                        {...omit(this.props, [
                            'controlClassName',
                            'tooltipClassName',
                            'tooltipDirection',
                            'tooltipMaxWidth',
                            'tooltipMessage',
                            'tooltipSize',
                            'tooltipType',
                        ])}
                        className={targetClassName}
                    />
                    <Tooltip
                        className={tooltipClassName}
                        direction={tooltipDirection || direction}
                        maxWidth={tooltipMaxWidth || maxWidth}
                        size={tooltipSize || size}
                        type={tooltipType || type}
                    >
                        {tooltipMessage}
                    </Tooltip>
                </div>
            );
        }
    }

    return Wrapper;
}
