import {parse as urlParse} from 'url';
import {includes, omit, random, uniq, find} from 'lodash';

import {
    CREATE_FORM_FIELD_CHANGE,
    REQUEST_CREATE_BETA,
    END_CREATE_BETA,
    BETA_CREATED,
    REQUEST_MIGRATION_FILES,
    RECEIVE_MIGRATION_FILES,
    RECEIVE_DB_IMAGES,
    RECEIVE_PRODUCTION_TAGS,
    SHOW_VALIDATION_ERRORS,
} from '../actions/createBetaFormActions';
import {RECEIVE_BETA_LIST} from '../actions/indexActions';
import dcs from '../dcs';

const SERVICES_URLS = {
    blackbox: {
        mock: 'https://blackbox-mock-partner.qloud.yandex-team.ru',
        real: 'http://blackbox-mimino.yandex.net',
    },
    yacotools: {
        mock: 'https://yacotools-mock-partner.qloud.yandex-team.ru',
        real: 'http://yacotools01et.yandex.ru',
    },
    bs: {
        mock: 'http://dev-partner2.ppc.yandex.ru:8099',
        real: 'http://bssoap-1.bssoap.pi-integration-qloud.bssoap.yabs-qabs-osc.stable.qloud-d.yandex.net',
    },
    adfox_host: {
        default: 'test.ui-dev.adfox.yandex.ru'
    }
};

const ubuntuVersions = ['bionic', 'precise']

const INITIAL_STATE = {
    errors: {},
    pending: false,

    dc: null,
    port: null,
    ubuntuVersion: 'bionic',
    backendBranch: 'trunk',
    frontendBranch: 'trunk',
    javaBranch: 'trunk',

    db: 'dev',
    selectedMigrationSqlFiles: ['misc.sql'],
    selectedMigrationScriptFiles: [],

    blackbox: SERVICES_URLS.blackbox.real,
    yacotools: SERVICES_URLS.yacotools.real,
    bs: SERVICES_URLS.bs.real,
    adfox_host: SERVICES_URLS.adfox_host.default,
    comment: '',
    ttl: '24',

    productionBackendTag: null,
    productionFrontendTag: null,

    dcs,
    ubuntuVersions,
    migrationSqlFiles: {
        myt: [],
        sas: [],
    },
    migrationScriptFiles: {
        myt: [],
        sas: [],
    },
    dbImage: null,
    dbImages: {
        myt: [],
        sas: [],
    },
    dbs: ['dev', 'ts', 'docker']
};

function _getInitialState() {
    const state = _updateStatePropertiesFromUrl(INITIAL_STATE, ['comment', 'backendBranch']);
    return Object.assign({}, state, {dc: state.dcs[random(state.dcs.length - 1)]});
}

function createBetaFormReducer(state = _getInitialState(), action) {
    const {type, dc} = action;

    switch (type) {
        case CREATE_FORM_FIELD_CHANGE: {
            const {fieldName, fieldValue} = action;
            let {errors} = state;

            if (errors && errors[fieldName]) {
                errors = Object.assign({}, errors, {
                    [fieldName]: null
                });
            }

            let {port, dbImage, selectedMigrationSqlFiles, selectedMigrationScriptFiles} = state;

            if (fieldName === 'dc') {
                port = null;
            }

            return Object.assign({}, state, {
                port,
                dbImage,
                selectedMigrationSqlFiles,
                selectedMigrationScriptFiles
            }, {
                [fieldName]: fieldValue,
                errors,
            });
        }

        case RECEIVE_BETA_LIST: {
            if (action.dc === state.dc) {
                const port = action.betaList
                    .filter(beta => beta.status === 'free')
                    .map(beta => beta.port)
                    .sort((a, b) => {
                        if (a < b) {
                            return -1;
                        } else if (b > a) {
                            return 1;
                        } else {
                            return 0;
                        }
                    })[0];
                return Object.assign({}, state, {port});
            } else {
                return state;
            }
        }

        case REQUEST_CREATE_BETA:
            return Object.assign({}, state, {
                pending: true
            });

        case END_CREATE_BETA:
            return Object.assign({}, state, {pending: false});

        case BETA_CREATED:
            return Object.assign({}, state, omit(INITIAL_STATE, [
                'dc',
                'migrationSqlFiles',
                'migrationScriptFiles',
                'dbImages'
            ]));

        case REQUEST_MIGRATION_FILES: {
            const migrationSqlFiles = Object.assign({}, state.migrationSqlFiles, {[dc]: null});
            const migrationScriptFiles = Object.assign({}, state.migrationScriptFiles, {[dc]: null});
            return Object.assign({}, state, {
                migrationSqlFiles,
                migrationScriptFiles,
                selectedMigrationSqlFiles: [],
                selectedMigrationScriptFiles: []
            });
        }

        case RECEIVE_MIGRATION_FILES: {
            const migrationSqlFiles = Object.assign({}, state.migrationSqlFiles, {[dc]: Array.isArray(action.migrationSqlFiles) ? action.migrationSqlFiles : []});
            const migrationScriptFiles = Object.assign({}, state.migrationScriptFiles, {[dc]: Array.isArray(action.migrationScriptFiles) ? action.migrationScriptFiles : []});
            const newState = Object.assign({}, state, {migrationSqlFiles, migrationScriptFiles});

            const selectedMigrationSqlFiles = _getSelectedFiles(newState, state.backendBranch, newState.migrationSqlFiles, newState.selectedMigrationSqlFiles);
            const selectedMigrationScriptFiles = _getSelectedFiles(newState, state.backendBranch, newState.migrationScriptFiles, newState.selectedMigrationScriptFiles);

            return Object.assign({}, newState, {selectedMigrationSqlFiles, selectedMigrationScriptFiles});
        }

        case RECEIVE_PRODUCTION_TAGS: {
            const {backendTag, frontendTag} = action;

            return Object.assign({}, state, {
                productionBackendTag: backendTag,
                productionFrontendTag: frontendTag
            });
        }

        case RECEIVE_DB_IMAGES: {
            const {dbImages} = action;
            const newDbImages = dbImages.reverse();
            const dbImagesNewState = Object.assign({}, state.dbImages, {[dc]: newDbImages});
            return Object.assign({}, state, {
                dbImages: dbImagesNewState
            });
        }

        case SHOW_VALIDATION_ERRORS: {
            const {errors} = action;
            return Object.assign({}, state, {errors});
        }

        default:
            return state;
    }
}

function _getSelectedFiles(state, backendBranch, migrationFiles, selectedMigrationFiles) {
    const {dc} = state;

    const availableFiles = migrationFiles[dc];

    if (!Array.isArray(availableFiles)) {
        return null;
    }

    const filesToAdd = availableFiles.filter(file => !includes(file, 'MANUAL'));

    return uniq((selectedMigrationFiles || []).concat(filesToAdd));
}

function _addNeedMigrationsError(errors) {
    const msg = "Не смогли найти нужный образ БД — обратись к автору бэкендной ветки";
    return Object.assign({}, errors, {dbImage: msg});
}

function _updateStatePropertiesFromUrl(state, propertyNameList) {
    return propertyNameList.reduce(
        (acc, propertyName) => Object.assign({}, _updateStatePropertyFromUrl(acc, propertyName)),
        state
    );
}

function _updateStatePropertyFromUrl(state, propertyName) {
    const propertyValue = _getParameterFromUrl(propertyName);
    if (propertyValue !== null) {
        return Object.assign({}, state, {[propertyName]: propertyValue});
    }
    return state;
}

function _getParameterFromUrl(parameterName) {
    const urlObject = urlParse(window.location.toString(), true);
    const {query = {}} = urlObject;
    return query[parameterName] || null;
}

export {SERVICES_URLS, createBetaFormReducer};
