'use strict';

const {chain, forEach, mapValues, reduce, sortBy} = require('lodash');
const {join, relative} = require('path');
const {print, saferequire} = require('tanker-kit').util;
const glob = require('glob').sync;
const rimraf = require('rimraf').sync;

const INDENT = 4;
const NOT_VISITED = 0;
const VISITED = 1;

module.exports = dispatcher;

/**
 * @public
 * @param  {object} options
 * @param  {string} options.data
 * @param  {object} options.info
 * @return {object} {[filepath]: string}
 */
function dispatcher({data, info}) {
    const {langs} = info.config.tanker;
    const {fprefix: desiredPath, root: rootPath, tankerMap} = info.config.paths;

    const keysMap = saferequire(tankerMap);

    if (!keysMap) {
        throw new Error('.tanker/map.json was not found. You should `tanker sync` first');
    }

    const existingI18nDirs = _existingKeysets('**/*.i18n/', rootPath, desiredPath);

    const relativeDesiredPath = relative(rootPath, desiredPath);
    const groupedMap = chain(keysMap)
        .values()
        .flatten()
        .filter(({path}) => path.startsWith(relativeDesiredPath))
        .groupBy(({path}) => path)
        .value();

    const json = JSON.parse(data);
    const result = reduce(groupedMap, (acc, keys, path) => {
        const baseI18nDir = path.replace(/(js|tsx?)$/, 'i18n/');

        if (existingI18nDirs[baseI18nDir]) {
            existingI18nDirs[baseI18nDir].mark = VISITED;
        }

        sortBy(keys, ({key, keyset}) => keyset + key).forEach(keyDesc => langs.forEach(lang => {
            const filepath = join(baseI18nDir, lang + '.json');
            const keyset = keyDesc.keyset;
            const blockName = keyset.substr(keyset.lastIndexOf(':') + 1);
            const data = json[lang][keyset];

            if (!acc[filepath]) {
                acc[filepath] = {};
            }

            if (!acc[filepath][blockName]) {
                acc[filepath][blockName] = {};
            }

            const {key, plural} = keyDesc;
            acc[filepath][blockName][key] = _formatKeyValue(key, data[key], plural, lang);
        }));

        return acc;
    }, {});

    const serializedResult = mapValues(result, value => JSON.stringify(value, null, INDENT));

    forEach(existingI18nDirs, ({mark, path}) => {
        if (mark === NOT_VISITED) {
            const abspath = join(rootPath, path);

            rimraf(abspath);
            print.info('Unused keyset directory was deleted: ' + path);
        }
    });

    return serializedResult;
}

/**
 * @private
 * @param  {string} pattern
 * @param  {string} rootPath
 * @return {object}
 */
function _existingKeysets(pattern, rootPath, desiredPath) {
    const paths = glob(pattern, {
        absolute: false,
        ignore: 'node_modules',
        root: rootPath,
    }).filter(path => path.startsWith(desiredPath));

    return paths.reduce((acc, path) => {
        acc[path] = {mark: NOT_VISITED, path};

        return acc;
    }, {});
}

/**
 * @private
 * @param  {string} key
 * @param  {string|string[]} translation
 * @param  {boolean} plural
 * @param  {string} lang
 * @return {string|string[]}
 */
function _formatKeyValue(key, translation, plural, lang) {
    if (plural === false) {
        return translation === '' ? key : translation;
    }

    if (translation.every(form => form === '')) {
        return translation.map(() => key);
    }

    switch (lang) {
    case 'en':
        return translation.slice(0, 2);
    case 'ru':
        return translation.slice(0, 3);
    case 'tr':
        return [translation[0]];
    }
}
