'use strict';

const {extname, relative, sep} = require('path');
const {noop} = require('lodash');
const babylon = require('@babel/parser');
const walk = require('acorn/dist/walk');

const serializeNode = require('./serializeNode');

const PLURAL_IDENT_NAME = 'count';

/**
 * @param  {object} options
 * @param  {string} options.data Содержимое найденного файла
 * @param  {string} options.path Путь к файлу
 * @param  {string} options.type Тип файла
 * @return {array}
 */
module.exports = function parse(options) {
    const keys = [];
    const isTS = (options.type === 'tsx' || options.type  === 'ts');

    const ast = babylon.parse(options.data, {
        plugins: [
            'classProperties',
            'estree',
            (isTS) ? 'typescript' : 'flow',
            'jsx',
        ],
        sourceType: 'module',
    });

    walk.recursive(ast, null, {
        CallExpression(node, st, c) {
            if (node.callee.name === '__') {
                const key = _formatExprArguments(node);
                keys.push(_formatKey(key, options.path));
            } else {
                c(node.callee, st);
                node.arguments.forEach(argument => c(argument, st));
            }
        },

        ClassProperty: noop,

        DeclareFunction(node, st, c) {
            c(node.id, st);
        },

        JSXClosingElement: noop,

        JSXAttribute(node, st, c) {
            c(node.name, st);

            // boolean jsx props don't have value
            if (node.value !== null) {
                c(node.value, st);
            }
        },

        JSXElement(node, st, c) {
            c(node.openingElement, st);
            node.children.forEach(child => c(child, st));
        },

        JSXExpressionContainer(node, st, c) {
            c(node.expression, st);
        },

        JSXIdentifier(node, st, c) {
            c(node, st, 'Identifier');
        },

        JSXOpeningElement(node, st, c) {
            node.attributes.forEach(attr => c(attr, st));
        },

        JSXSpreadAttribute(node, st, c) {
            c(node.argument, st);
        },

        JSXText: noop,

        File(node, st, c) {
            c(node.program, st);
        },

        TypeAlias: noop,

        TypeAnnotation: noop,

        TypeCastExpression: noop,
    });

    return keys;
};

/**
 * @private
 * @param  {object} expr
 * @return {object}
 */
function _formatExprArguments(expr) {
    const args = expr.arguments;
    let keyExpr;
    let keysetExpr;

    if (args.length === 1) {
        keyExpr = args[0];
    } else if (args.length === 2) {
        const keyOrArgs = args[1];

        if (keyOrArgs.type === 'ObjectExpression') {
            keyExpr = args[0];
        } else {
            keysetExpr = args[0];
            keyExpr = args[1];
        }

    } else if (args.length === 3) {
        keysetExpr = args[0];
        keyExpr = args[1];
    } else {
        throw new Error('Unsupported expression');
    }

    // switch (args.length) {
    // case 2:
    //     keysetExpr = args[0];
    //     keyExpr = args[1];
    //     break;
    // case 1:
    //     keyExpr = args[0];
    //     break;
    // default:
    //     throw new Error('Unsupported expression');
    // }

    const key = serializeNode(keyExpr);
    const keyset = keysetExpr && serializeNode(keysetExpr) || null;
    const plural = (
        Array.isArray(keyExpr.expressions) &&
        keyExpr.expressions.some(node => node.type === 'Identifier' &&
        node.name === PLURAL_IDENT_NAME)
    ) || (
        key.includes('{' + PLURAL_IDENT_NAME + '}')
    );

    return {plural, keyset, key};
}

/**
 * @private
 * @param  {object}      options
 * @param  {string}      options.key
 * @param  {string|null} options.keyset
 * @param  {boolean}     options.plural
 * @param  {string}      filepath
 * @return {object}
 */
function _formatKey({key, keyset, plural}, filepath) {
    const contextKeyset = !keyset
        ? _formatKeyset(filepath)
        : keyset;

    return {
        key,
        keyset: contextKeyset,
        type: 'component',
        hash: _buildKeyId(contextKeyset, key, filepath),
        plural,
    };
}

/**
 * @private
 * @param  {string} filepath
 * @return {string}
 */
function _formatKeyset(filepath) {
    if (filepath.startsWith('src')) {
        filepath = 'form' + sep + relative('src', filepath);
    }

    return filepath.replace(extname(filepath), '').replace(new RegExp(sep, 'g'), ':');
}

/**
 * @private
 * @param  {string} contextKeyset
 * @param  {string} key
 * @param  {string} filepath
 * @return {string}
 */
function _buildKeyId(contextKeyset, key, filepath) {
    return Buffer.from(`${contextKeyset}|${key} ${filepath}`).toString('base64');
}
