/* global __dirname */
'use strict';

const fs = require('fs');
const path = require('path');
const {promisify} = require('util');

const stat = promisify(fs.stat);
const readFile = promisify(fs.readFile);

const langs = ['en', 'ru'];

function insertInitPatch(code, initPatch) {
    const strictStmt = code.match(/['"]use strict['"];/);
    if (!strictStmt) {
        return initPatch + code;
    }
    const insertAt = strictStmt.index + strictStmt[0].length;
    return code.substring(0, insertAt) + '\n' + initPatch + code.substring(insertAt);
}

function getClassWithAsset(asset) {
    class MyAsset extends asset {
        async getTranslations() {
            if (this.relativeName.startsWith('../node_modules')) {
                return null;
            }

            const strings = {};

            const assetDir = path.dirname(this.name);
            const moduleName = this.basename.split('.').slice(0, -1).join('.');

            const stringModules = langs.reduce((acc, lang) => {
                acc[lang] = path.join(assetDir, `${moduleName}.i18n`, `${lang}.json`);
                return acc;
            }, {});

            for (const lang of langs) {
                try {
                    const moduleStat = await stat(stringModules[lang]);
                    strings[lang] = moduleStat.isFile()
                        ? JSON.parse(await readFile(stringModules[lang], { encoding: 'utf8' }))
                        : {};
                } catch (err) {
                    strings[lang] = {};
                }
            }

            const hasTranslations = langs.reduce(
                (acc, lang) =>
                    Object.keys(strings[lang]).length > 0 || acc,
                false,
            );

            if (!hasTranslations) {
                return null;
            }

            return JSON.stringify(strings);
        }

        async appendI18nCode(code) {
            const strings = await this.getTranslations();

            // TODO: Rework to provide translation function even when strings are absent
            if (!strings) {
                const hasI18nCall = code.indexOf('__(') !== -1;

                const dummyCode = `;
                    // aboba
                    function __(string) {
                        return string;
                    };
                `;

                return hasI18nCall
                    ? insertInitPatch(code, dummyCode)
                    : code;
            }

            const moduleName = this.basename.split('.').slice(0, -1).join('.');
            const projectRootPath = path.resolve(__dirname, '../');
            const absI18nModulePath = path.join(projectRootPath, 'src/lib/i18n.js');
            let i18nModulePath = path.relative(path.dirname(this.name), absI18nModulePath);

            if (!i18nModulePath.startsWith('../')) {
                i18nModulePath = './' + i18nModulePath;
            }

            const i18nCode = `;
                const i18n = require('${i18nModulePath}');
                i18n.addStrings(${strings});

                function __(...args) {
                    if (args.length === 0) {
                        return '';
                    } else if (args.length === 1) {
                        return i18n.translate('${moduleName}', args[0]);
                    } else if (args.length === 2) {
                        if (typeof args[1] === 'object') {
                            return i18n.translate('${moduleName}', args[0], args[1]);
                        } else {
                            return i18n.translate(args[0], args[1]);
                        }
                    }

                    return i18n.translate(args[0], args[1], args[2]);
                };
            `;

            return insertInitPatch(code, i18nCode);
        }

        async load() {
            const contents = await super.load();
            return this.appendI18nCode(contents);
        }
    }

    return MyAsset;
}




module.exports = getClassWithAsset;
