// @flow
'use strict';

import type {
    BranchDataT,
    FiasT,
    OfferDataT,
    TreeDataT,
    UserDataT,
} from '../types/state';

export type ActionT<T, P> = {
    type: T,
    payload: P,
};

function action<T, P>(
    type: T,
    payload: P,
): ActionT<T, P> {
    return {
        type,
        payload,
    };
}

export type FetchTreeActionT = ActionT<'FETCH_TREE_ACTION', null>;
export const FETCH_TREE_ACTION = 'FETCH_TREE_ACTION';
export const fetchTree = (): FetchTreeActionT =>
    action(FETCH_TREE_ACTION, null);

export type FieldBlurActionT = ActionT<'FIELD_BLUR_ACTION', {id: string}>;
export const FIELD_BLUR_ACTION = 'FIELD_BLUR_ACTION';
export const fieldBlur = (id: string): FieldBlurActionT =>
    action(FIELD_BLUR_ACTION, {id});

export type FieldChangeActionT = ActionT<'FIELD_CHANGE_ACTION', {id: string, value: string}>;
export const FIELD_CHANGE_ACTION = 'FIELD_CHANGE_ACTION';
export const fieldChange = (id: string, value: string): FieldChangeActionT =>
    action(FIELD_CHANGE_ACTION, {id, value});

export type FieldFocusActionT = ActionT<'FIELD_FOCUS_ACTION', {id: string}>;
export const FIELD_FOCUS_ACTION = 'FIELD_FOCUS_ACTION';
export const fieldFocus = (id: string): FieldFocusActionT =>
    action(FIELD_FOCUS_ACTION, {id});

export type InitStoreActionT = ActionT<'INIT_STORE_ACTION', {defaults: {[fieldId: string]: any}}>;
export const INIT_STORE_ACTION = 'INIT_STORE_ACTION';
export const initStore = (defaults: {[fieldId: string]: any}): InitStoreActionT =>
    action(INIT_STORE_ACTION, {defaults});

type ReceiveUserDataPayloadT = {data: UserDataT | {is_logged: false}};
export type ReceiveUserDataActionT = ActionT<'RECEIVE_USER_DATA_ACTION', ReceiveUserDataPayloadT>;
export const RECEIVE_USER_DATA_ACTION = 'RECEIVE_USER_DATA_ACTION';
export const receiveUserData = (data: ReceiveUserDataPayloadT): ReceiveUserDataActionT =>
    action(RECEIVE_USER_DATA_ACTION, data);

type ReceiveTreePayloadT = {data: TreeDataT};
export type ReceiveTreeActionT = ActionT<'RECEIVE_TREE_ACTION', ReceiveTreePayloadT>;
export const RECEIVE_TREE_ACTION = 'RECEIVE_TREE_ACTION';
export const receiveTree = (data: ReceiveTreePayloadT): ReceiveTreeActionT =>
    action(RECEIVE_TREE_ACTION, data);

export type FetchBranchActionT = ActionT<'FETCH_BRANCH_ACTION', {branchId: string}>;
export const FETCH_BRANCH_ACTION = 'FETCH_BRANCH_ACTION';
export const fetchBranch = (branchId: string): FetchBranchActionT =>
    action(FETCH_BRANCH_ACTION, {branchId});

type ReceiveBranchPayloadT = {data: BranchDataT};
export type ReceiveBranchActionT = ActionT<'RECEIVE_BRANCH_ACTION', ReceiveBranchPayloadT | null>;
export const RECEIVE_BRANCH_ACTION = 'RECEIVE_BRANCH_ACTION';
export const receiveBranch = (data: ReceiveBranchPayloadT | null): ReceiveBranchActionT =>
    action(RECEIVE_BRANCH_ACTION, data);

export type BranchChangeActionT = ActionT<'BRANCH_CHANGE_ACTION', {branchId: string | null}>;
export const BRANCH_CHANGE_ACTION = 'BRANCH_CHANGE_ACTION';
export const branchChange = (branchId: string | null): BranchChangeActionT =>
    action(BRANCH_CHANGE_ACTION, {branchId});

export type CountryChangeActionT = ActionT<'COUNTRY_CHANGE_ACTION', {countryId: string}>;
export const COUNTRY_CHANGE_ACTION = 'COUNTRY_CHANGE_ACTION';
export const countryChange = (countryId: string): CountryChangeActionT =>
    action(COUNTRY_CHANGE_ACTION, {countryId});

export type SubmitActionT = ActionT<'SUBMIT_ACTION', null>;
export const SUBMIT_ACTION = 'SUBMIT_ACTION';
export const submit = (): SubmitActionT =>
    action(SUBMIT_ACTION, null);

export type LockPendingActionT = ActionT<'LOCK_PENDING_ACTION', null>;
export const LOCK_PENDING_ACTION = 'LOCK_PENDING_ACTION';
export const lockPending = (): LockPendingActionT =>
    action(LOCK_PENDING_ACTION, null);

export type UnlockPendingActionT = ActionT<'UNLOCK_PENDING_ACTION', null>;
export const UNLOCK_PENDING_ACTION = 'UNLOCK_PENDING_ACTION';
export const unlockPending = (): UnlockPendingActionT =>
    action(UNLOCK_PENDING_ACTION, null);

export type SerializeFieldsActionT = ActionT<'SERIALIZE_FIELDS_ACTION', null>;
export const SERIALIZE_FIELDS_ACTION = 'SERIALIZE_FIELDS_ACTION';
export const serializeFields = (): SerializeFieldsActionT =>
    action(SERIALIZE_FIELDS_ACTION, null);

export type SetFieldErrorActionT = ActionT<'SET_FIELD_ERROR_ACTION', {
    id: string,
    error: string | {[subfieldId: string]: string} | null,
}>;
export const SET_FIELD_ERROR_ACTION = 'SET_FIELD_ERROR_ACTION';
export const setFieldError = (
    id: string,
    error: string | {[subfieldId: string]: string} | null,
): SetFieldErrorActionT =>
    action(SET_FIELD_ERROR_ACTION, {id, error});

export type SetFieldFocusActionT = ActionT<'SET_FIELD_FOCUS_ACTION', {id: string, focus: boolean}>;
export const SET_FIELD_FOCUS_ACTION = 'SET_FIELD_FOCUS_ACTION';
export const setFieldFocus = (id: string, focus: boolean): SetFieldFocusActionT =>
    action(SET_FIELD_FOCUS_ACTION, {id, focus});

export type SetFieldPendingActionT = ActionT<'SET_FIELD_PENDING_ACTION', {id: string, pending: boolean}>;
export const SET_FIELD_PENDING_ACTION = 'SET_FIELD_PENDING_ACTION';
export const setFieldPending = (id: string, pending: boolean): SetFieldPendingActionT =>
    action(SET_FIELD_PENDING_ACTION, {id, pending});

type ReceiveBackendErrorsPayloadT = {
    errors: {[fieldId: string]: Array<{description: string}>},
};
export type ReceiveBackendErrorsActionT = ActionT<'RECEIVE_BACKEND_ERRORS_ACTION', ReceiveBackendErrorsPayloadT>;
export const RECEIVE_BACKEND_ERRORS_ACTION = 'RECEIVE_BACKEND_ERRORS_ACTION';
export const receiveBackendErrors = (payload: ReceiveBackendErrorsPayloadT): ReceiveBackendErrorsActionT =>
    action(RECEIVE_BACKEND_ERRORS_ACTION, payload);

export type ScrollToActionT = ActionT<'SCROLL_TO_ACTION', {id: string | null}>;
export const SCROLL_TO_ACTION = 'SCROLL_TO_ACTION';
export const scrollTo = (id: string | null): ScrollToActionT =>
    action(SCROLL_TO_ACTION, {id});

export type PushErrorActionT = ActionT<'PUSH_ERROR_ACTION', {error: Error}>;
export const PUSH_ERROR_ACTION = 'PUSH_ERROR_ACTION';
export const pushError = (error: Error): PushErrorActionT =>
    action(PUSH_ERROR_ACTION, {error});

export type RemoveErrorActionT = ActionT<'REMOVE_ERROR_ACTION', {id: string}>;
export const REMOVE_ERROR_ACTION = 'REMOVE_ERROR_ACTION';
export const removeError = (id: string): RemoveErrorActionT =>
    action(REMOVE_ERROR_ACTION, {id});

type ReceiveOfferPayloadT = {data: OfferDataT};
export type ReceiveOfferActionT = ActionT<'RECEIVE_OFFER_ACTION', ReceiveOfferPayloadT>;
export const RECEIVE_OFFER_ACTION = 'RECEIVE_OFFER_ACTION';
export const receiveOffer = (data: ReceiveOfferPayloadT): ReceiveOfferActionT =>
    action(RECEIVE_OFFER_ACTION, data);

type ReceiveOfferHeadingsPayloadT = {headings: Array<string>};
export type ReceiveOfferHeadingsActionT = ActionT<'RECEIVE_OFFER_HEADINGS_ACTION', ReceiveOfferHeadingsPayloadT>;
export const RECEIVE_OFFER_HEADINGS_ACTION = 'RECEIVE_OFFER_HEADINGS_ACTION';
export const receiveOfferHeadings = (data: ReceiveOfferHeadingsPayloadT): ReceiveOfferHeadingsActionT =>
    action(RECEIVE_OFFER_HEADINGS_ACTION, data);

export type ChangeLocaleActionT = ActionT<'CHANGE_LOCALE_ACTION', {locale: string}>;
export const CHANGE_LOCALE_ACTION = 'CHANGE_LOCALE_ACTION';
export const changeLocale = (locale: string): ChangeLocaleActionT =>
    action(CHANGE_LOCALE_ACTION, {locale});

export type FileUploadFieldAddActionT = ActionT<'FILE_UPLOAD_FIELD_ADD_ACTION', {
    fieldId: string,
    fileId: string,
    file: File,
}>;
export const FILE_UPLOAD_FIELD_ADD_ACTION = 'FILE_UPLOAD_FIELD_ADD_ACTION';
export const fileUploadFieldAdd = (fieldId: string, fileId: string, file: File): FileUploadFieldAddActionT =>
    action(FILE_UPLOAD_FIELD_ADD_ACTION, {fieldId, fileId, file});

export type FileUploadFieldDeleteActionT = ActionT<'FILE_UPLOAD_FIELD_DELETE_ACTION', {
    fieldId: string,
    fileId: string,
}>;
export const FILE_UPLOAD_FIELD_DELETE_ACTION = 'FILE_UPLOAD_FIELD_DELETE_ACTION';
export const fileUploadFieldDelete = (fieldId: string, fileId: string): FileUploadFieldDeleteActionT =>
    action(FILE_UPLOAD_FIELD_DELETE_ACTION, {fieldId, fileId});

export type FileUploadFieldFinishedActionT = ActionT<'FILE_UPLOAD_FIELD_FINISHED_ACTION', {
    fieldId: string,
    fileId: string,
    ok: boolean,
    key: string | null
}>;
export const FILE_UPLOAD_FIELD_FINISHED_ACTION = 'FILE_UPLOAD_FIELD_FINISHED_ACTION';
export const fileUploadFieldFinished = (
    fieldId: string,
    fileId: string,
    ok: boolean,
    key: string | null
): FileUploadFieldFinishedActionT =>
    action(FILE_UPLOAD_FIELD_FINISHED_ACTION, {fieldId, fileId, ok, key});

export type FetchFiasActionT = ActionT<'FETCH_FIAS_ACTION', {guid: string | null, fieldId: string | null}>;
export const FETCH_FIAS_ACTION = 'FETCH_FIAS_ACTION';
export const fetchFias = (guid: string | null, fieldId?: string): FetchFiasActionT =>
    action(FETCH_FIAS_ACTION, {guid, fieldId: fieldId || null});

export type ReceiveFiasActionT = ActionT<'RECEIVE_FIAS_ACTION', {guid: string, data: Array<FiasT>}>;
export const RECEIVE_FIAS_ACTION = 'RECEIVE_FIAS_ACTION';
export const receiveFias = (guid: string, data: Array<FiasT>): ReceiveFiasActionT =>
    action(RECEIVE_FIAS_ACTION, {guid, data});
