// @flow
'use strict';

const {assign} = Object;

import {pick} from 'lodash';
import * as React from 'react';

import {Input, Select} from 'teatime-components';
import {FormField} from '../FormField/FormField';

import css from './Address.css';

import type {
    AddressT,
    FiasDataT,
    FiasT,
} from '../../types/state';

type PropsT = {
    disabled: boolean,
    displayError: Object | string,
    fias: FiasDataT,
    isLegalAddress: boolean,
    name: string,
    onBlur?: () => void,
    onChange?: (value: AddressT) => void,
    onFetchFias?: (guid: string | null) => void,
    onFocus?: () => void,
    pending: boolean,
    value: AddressT,
};

const REGION_LEVEL = '1';
const STREET_LEVEL = '7';

const LEVEL_NAMES = {
    '1': __('Region'),
    '2': __('Autonomous region'),
    '3': __('Area'),
    '4': __('City'),
    '5': __('Intracity area'),
    '6': __('Locality'),
    '7': __('Street'),
};

export class Address extends React.Component<PropsT> {
    static defaultProps: PropsT = {
        disabled: false,
        displayError: {},
        fias: {},
        isLegalAddress: false,
        name: 'address',
        pending: false,
        value: {
            apartment: null,
            building: null,
            guid: '',
            house: '',
            levels: {},
            office: null,
            structure: null,
            zipCode: '',
        },
    }

    constructor(props: PropsT) {
        super(props);

        this._onChange = this._onChange.bind(this);
    }

    _onChange: (fieldId: string, value: string | Object) => void

    _onChange(fieldId: string, fieldValue: string | Object): void {
        const {value, onChange} = this.props;

        if (!onChange) {
            return;
        }

        const nextValue = assign({}, value, {[fieldId]: fieldValue});

        onChange(nextValue);
    }

    _onFieldChange(level: string, val: Object): void {
        const {value, onFetchFias} = this.props;
        const {levels = {}} = value;

        const levelsToInclude = Object.keys(levels).filter(savedLevel => Number(savedLevel) < Number(level));

        this._onChange(
            'levels',
            assign({}, pick(levels, levelsToInclude), { [level]: val.value }),
        );

        if (onFetchFias && level !== STREET_LEVEL) {
            onFetchFias(val.value);
        }
    }

    componentDidMount(): void {
        const {onFetchFias} = this.props;

        if (!onFetchFias) {
            return;
        }

        onFetchFias(null);
    }

    render(): React.Element<*> {
        return (
            <div>
                {this._renderFields()}
                {this._renderGroupedFields()}
                {this._renderZipCode()}
            </div>
        );
    }

    _fiasItemsToOptions(
        items: Array<FiasT>,
        level: string,
    ): Array<{label: string, value: string | null}> {
        if (!items) {
            return [];
        }

        return [
            {label: '—', value: null},
        ].concat(
            items
                .map(item => ({
                    label: (level === STREET_LEVEL && item.shortName ? `${item.shortName} ` : '') + item.formalName,
                    value: item.guid }))
                .sort((a, b) => a.label > b.label ? 1 : -1)
        );
    }

    _searchEngine(searchVal: string, selectItem: string): boolean {
        if (typeof searchVal !== 'string' || typeof selectItem !== 'string') {
            return false;
        }

        // е и ё при поиске должны считаться одинаковыми
        const processValue = str => str.toLowerCase().replace(/ё/g, 'е');
        return processValue(selectItem).includes(processValue(searchVal));
    }

    _getItemsByLevel(fiasData: FiasDataT, levels: Object): {[level: string]: Array<FiasT>} {
        const sortedLevels = Object.keys(levels).sort();
        const result = {};

        result[REGION_LEVEL] = fiasData.ROOT;

        for (const level of sortedLevels) {
            const guid = levels[level];
            const fias = fiasData[guid];

            if (!fias) {
                continue;
            }

            const levelsToClear = Object.keys(result).filter(savedLevel => Number(savedLevel) > Number(level));

            for (const level of levelsToClear) {
                delete result[level];
            }

            for (const item of fias) {
                const {objLevel} = item;

                if (!result.hasOwnProperty(objLevel)) {
                    result[objLevel] = [];
                }

                result[objLevel].push(item);
            }
        }

        return result;
    }

    _renderFields(): React.Element<*> {
        const {fias, value} = this.props;
        const {levels = {}} = value;

        const itemsByLevel = this._getItemsByLevel(fias, levels);
        const fields = Object.keys(itemsByLevel).map(level => this._renderLevel(level, itemsByLevel[level]));

        return (
            <div>
                {fields}
            </div>
        );
    }

    _renderLevel(level: string, items: Array<FiasT>): React.Element<*> {
        const {disabled, displayError, onBlur, onFocus, pending, value} = this.props;
        const {levels = {}} = value;
        const guid = levels[level] || null;

        const levelName = LEVEL_NAMES[level] || null;

        const options = this._fiasItemsToOptions(items, level);

        const errors = typeof displayError === 'string'
            ? {}
            : displayError || {};

        const error = errors[level];

        return (
            <FormField
                key={`fias-${level}`}
                displayName={levelName}
                displayError={error}
            >
                <Select
                    className={css.input}
                    disabled={disabled || pending}
                    name={`address-${level}`}
                    onBlur={onBlur}
                    onChange={(event, val) => this._onFieldChange(level, val)}
                    onFocus={onFocus}
                    options={options}
                    searchEngine={this._searchEngine}
                    searchable={true}
                    size='l'
                    value={guid}
                />
            </FormField>
        );
    }

    _renderZipCode(): React.Element<*> | null {
        const {disabled, displayError, onBlur, onFocus, pending, value} = this.props;
        const fieldId = 'zipCode';
        const fieldValue = value[fieldId];

        const {levels = {}} = value;
        const streetLevel = levels[STREET_LEVEL];

        if (!streetLevel) {
            return null;
        }

        const errors = typeof displayError === 'string'
            ? {}
            : displayError || {};

        const error = errors.zipCode;

        return (
            <FormField
                key='zipCode'
                displayName={__('Zip code')}
                displayError={error}
                componentClassName={css.zipCode}
            >
                <Input
                    className={css.zipCode}
                    disabled={disabled || pending}
                    name={`address-${fieldId}`}
                    onBlur={onBlur}
                    onChange={(event, val) => this._onChange(fieldId, val.value)}
                    onFocus={onFocus}
                    value={fieldValue}
                />
            </FormField>
        );
    }

    _renderGroupedFields(): React.Element<*> | null {
        const {isLegalAddress, value} = this.props;
        const {levels = {}} = value;
        const streetLevel = levels[STREET_LEVEL];

        if (!streetLevel) {
            return null;
        }

        const fields = [
            {id: 'house', name: __('House')},
            {id: 'building', name: __('Building')},
            {id: 'structure', name: __('Structure')},
        ];

        if (isLegalAddress) {
            fields.push({id: 'office', name: __('Office')});
        } else {
            fields.push({id: 'apartment', name: __('Apartment')});
        }

        return (
            <div className={css.groupedFields}>
                {fields.map(({id, name}) => this._renderGroupedField(id, name))}
            </div>
        );
    }

    _renderGroupedField(fieldId: string, fieldName: string): React.Element<*> {
        const {disabled, displayError, onBlur, onFocus, name, pending, value} = this.props;
        const fieldValue = value[fieldId];

        const errors = typeof displayError === 'string'
            ? {}
            : displayError || {};

        const error = errors[fieldId];

        return (
            <FormField
                className={css.groupedField}
                componentClassName={css.groupedInput}
                key={`${name}-${fieldId}`}
                displayName={fieldName}
                displayError={error}
            >
                <Input
                    disabled={disabled || pending}
                    name={`address-${fieldId}`}
                    onBlur={onBlur}
                    onChange={(event, val) => this._onChange(fieldId, val.value)}
                    onFocus={onFocus}
                    value={fieldValue}
                />
            </FormField>
        );
    }
}
