// @flow
'use strict';

const {assign} = Object;
import * as React from 'react';
import classnames from 'classnames';

import {Check, Input, Link, Spin} from 'teatime-components';
import {FormField} from '../FormField/FormField';
import {OfferCheck} from '../OfferCheck/OfferCheck';

import css from './AdfoxAccount.css';

import type {AdfoxAccountT as ValueT} from '../../types/state';

import {
    WRONG_ADFOX_CREDENTIALS_ERROR_TOKEN,
} from '../../types/fields';

const RESTORE_PASSWORD_URL = 'https://auth.adfox.ru/password/recovery/';
const PAID_SERVICES_OFFER_URL = 'https://yandex.ru/legal/adfox_tos_pvd/';

type PropsT = {
    disabled: boolean,
    displayError: Object | null,
    value: ValueT,
    hint: Object,
    onBlur?: () => void,
    onChange?: (value: ValueT) => void,
    onFocus?: () => void,
    pending: boolean,
    adfoxOffer: boolean,
    password?: string,
    login?: string,
};

type StateT = {
    // Login and password values at last time the "adfox offer" was checked
    // Introduced to detect no change in credentials inputs after use
    // stops typing (and input blurs away).
    checkedLogin: string,
    checkedPassword: string,
};

export class AdfoxAccount extends React.Component<PropsT, StateT> {
    static defaultProps: PropsT = {
        adfoxOffer: false,
        displayError: null,
        disabled: false,
        hint: {},
        value: {
            hasAccount: false,
            hasAdfoxPaid: false,
            login: '',
            password: '',
        },
        pending: false,
    };

    _loginInput: Element;
    _passwordInput: Element;

    constructor(props: PropsT) {
        super(props);

        this._onCheckChange = this._onCheckChange.bind(this);
        this._onInputFieldChange = this._onInputFieldChange.bind(this);
        this._onInputFieldBlur = this._onInputFieldBlur.bind(this);
        this._onOfferCheckChange = this._onOfferCheckChange.bind(this);

        const {adfoxOffer, login, password} = this.props;

        this.state = {
            checkedLogin: adfoxOffer ? String(login) : '',
            checkedPassword: adfoxOffer ? String(password) : '',
        };
    }

    _onCheckChange: (event: Event, value: {name: string, checked: boolean}) => void;
    _onInputFieldChange: (event: Event, value: {name: string, value: string}) => void;
    _onOfferCheckChange: (event: Event, value: {name: string, checked: boolean}) => void;
    _onInputFieldBlur: () => void;

    _onCheckChange(event: Event, value: {name: string, checked: boolean}): void {
        const {value: propsValue, onBlur, onChange, onFocus} = this.props;
        const {checked: fieldChecked} = value;

        if (!onChange || !onFocus || !onBlur) {
            return;
        }

        let patch: $Shape<ValueT> = {
            hasAccount: fieldChecked,
            hasAdfoxPaid: propsValue.hasAdfoxPaid,
        };

        if (fieldChecked) {
            patch = Object.assign(patch, {
                login: propsValue.login,
                password: propsValue.password,
            });
        }

        onFocus();
        onChange(patch);
        onBlur();
    }

    _onInputFieldBlur() {
        const {onBlur, onChange, value} = this.props;
        const {login, password} = value;
        const {checkedLogin, checkedPassword} = this.state;

        const haveAccountDataChanged = login !== checkedLogin || password !== checkedPassword;

        if (haveAccountDataChanged && onChange) {
            // Throw away adfox_has_paid token
            const updatedProps = {...value, hasAdfoxPaid: false};
            onChange(updatedProps);
        }

        if (onBlur) {
            onBlur();
        }
    }

    _onInputFieldChange(event: Event, value: {name: string, value: string}): void {
        const {value: propsValue, onChange} = this.props;
        const {name: fieldName, value: fieldValue} = value;

        if (!onChange) {
            return;
        }

        const updatedProps = {
            ...propsValue,
            [fieldName]: fieldValue,
        };

        onChange(updatedProps);
    }

    _rememberCheckedCredentials() {
        const {
            value: {
                login,
                password,
            },
        } = this.props;

        this.setState({
            checkedLogin: login,
            checkedPassword: password,
        });
    }

    _onOfferCheckChange(event: Event, value: {name: string, checked: boolean}): void {
        const {value: propsValue, onBlur, onChange, onFocus} = this.props;
        const {checked: fieldChecked} = value;

        if (fieldChecked) {
            this._rememberCheckedCredentials();
        }

        if (!onChange || !onFocus || !onBlur) {
            return;
        }

        onFocus();
        onChange(assign({}, propsValue, {adfoxOffer: fieldChecked}));
        onBlur();
    }

    componentDidUpdate(prevProps: PropsT) {
        const {
            value: {
                hasAdfoxPaid: prevHasAdfoxPaid,
            },
        } = prevProps;
        const {
            value: {
                adfoxOffer,
                hasAdfoxPaid,
            },
        } = this.props;

        if (hasAdfoxPaid && !prevHasAdfoxPaid && adfoxOffer) {
            // Case: user first typed some account credentials, received adfox_has_paid token,
            // checked adfox_offer, stepped back to edit credentials data.
            // After they finish editing:
            // 1) adfox_has_paid token is thrown away, adfox_offer checkbox become hidden,
            // 2) adfox_account is validated without adfox_offer
            // 3) if validation response contains adfox_has_paid token:
            // 3.1) it is remembered,
            // 3.2) adfox_offer checkbox already checked become visible,
            // 3.3) validation request is repeated with adfox_offer flag set,
            // 4) any errors from last validation are displayed as usual

            this._rememberCheckedCredentials();

            const {onBlur} = this.props;
            if (onBlur) {
                // Trigger validation request with adfox_offer set on
                onBlur();
            }
        }
    }

    render(): React.Node {
        const {disabled, displayError, onFocus, value} = this.props;
        const {hasAccount, hasAdfoxPaid, login = '', password = ''} = value;

        const errors = displayError || {};
        const {errorToken} = errors;

        const commonProps = {
            className: css.input,
            disabled: disabled || !hasAccount,
            onBlur: this._onInputFieldBlur,
            onChange: this._onInputFieldChange,
            onFocus,
        };

        const showPasswordRestoreLink =
            errorToken === WRONG_ADFOX_CREDENTIALS_ERROR_TOKEN;

        const showPaidServicesOfferCheck = hasAccount && hasAdfoxPaid;

        return (
            <div>
                <FormField
                    className={css.fieldContainer}
                    key='hasAccount'
                    displayError={errors.hasAccount}
                >
                    <Check
                        checked={hasAccount}
                        className={css.input}
                        disabled={disabled}
                        label={__('I have ADFOX account')}
                        name='hasAccount'
                        onChange={this._onCheckChange}
                        size='l'
                    />
                </FormField>
                <div className={classnames(css.subTextBase, css.checkSubText)}>
                    {__('New account will be created if you don\'t have one.')}
                </div>
                <div className={css.groupedFields}>
                    <FormField
                        key='login'
                        displayName={__('Login')}
                        displayError={errors.login}
                        className={css.groupedField}
                        componentClassName={css.groupedInput}
                    >
                        <Input
                            name='login'
                            value={login}
                            {...commonProps}
                        />
                    </FormField>
                    <FormField
                        key='password'
                        displayName={__('Password')}
                        displayError={errors.password}
                        className={css.groupedField}
                        componentClassName={css.groupedInput}
                    >
                        <Input
                            name='password'
                            value={password}
                            type='password'
                            {...commonProps}
                        />
                    </FormField>
                </div>
                <div className={classnames(css.error)}>
                    {
                        commonProps.disabled
                            ? null
                            : this._renderErrorMessages(
                                displayError,
                                showPasswordRestoreLink,
                                showPaidServicesOfferCheck
                            )
                    }
                </div>
                <div className={classnames(
                    css.subTextBase,
                    css.credentialsSubText,
                    {[css.hidden]: !hasAccount}
                )}>
                    {__('The offer will apply to the linked ADFOX account.')}
                </div>
            </div>
        );
    }

    _renderErrorMessages(
        displayError: Object | null,
        showPasswordRestoreLink: boolean,
        showPaidServicesOfferCheck: boolean,
    ) {
        const {pending: isPendingState} = this.props;

        return isPendingState
            ? (
                <div className={css.spin}>
                    <Spin/>
                </div>)
            : (
                <div>
                    {displayError && displayError.text}
                    {showPasswordRestoreLink && this._renderPasswordRestoreLink()}
                    {showPaidServicesOfferCheck && this._renderPaidServicesOfferCheck()}
                </div>
            );
    }

    _renderPasswordRestoreLink() {
        return (
            <div>
                <Link
                    className={classnames(css.active, css.credentialsSubText)}
                    size='s'
                    href={RESTORE_PASSWORD_URL}
                    target='_blank'
                >
                    {__('Restore password')}
                </Link>
            </div>
        );
    }

    _renderPaidServicesOfferCheck() {
        const {value} = this.props;
        const {adfoxOffer = false, hasAccount} = value;

        const message = __('I have read and accepted [linkStart]Terms of Offer[linkEnd] for "Targeting materials by gender, age and income" service.');

        return (
            <OfferCheck
                checked={adfoxOffer}
                className={classnames(css.input, css.credentialsSubText)}
                disabled={!hasAccount}
                linkSize='s'
                message={message}
                name='adfoxOffer'
                offerUrl={PAID_SERVICES_OFFER_URL}
                onChange={this._onOfferCheckChange}
            />
        );
    }
}
