// @flow
'use strict';

const {assign} = Object;
import * as React from 'react';

import {Input, Radio} from 'teatime-components';
import {FormField} from '../FormField/FormField';

import type {
    BikWithAccountOrYandexMoneyT as ValueT,
    BikWithAccountOrYandexMoneyModeT as ModeT,
} from '../../types/state';

import css from './BikWithAccountOrYandexMoney.css';

const BANK_MODE = 'bank';
const YANDEX_MONEY_MODE = 'yandexMoney';

type PropsT = {
    disabled: boolean,
    displayError: Object,
    value: ValueT,
    hint: Object,
    onBlur?: () => void,
    onChange?: (value: ValueT) => void,
    onFocus?: () => void,
};

const RADIO_NAME = 'modeRadio';

const BANK_DEFAULTS = {
    bik: '',
    account: '',
};

const YANDEX_MONEY_DEFAULTS = {
    yandexMoney: '',
};

export class BikWithAccountOrYandexMoney extends React.Component<PropsT> {
    static defaultProps: PropsT = {
        displayError: {},
        disabled: false,
        hint: {},
        value: {
            mode: BANK_MODE,
            fields: BANK_DEFAULTS,
        },
    }

    constructor(props: PropsT) {
        super(props);

        this._onFieldChange = this._onFieldChange.bind(this);
        this._onRadioChange = this._onRadioChange.bind(this);
    }

    _onFieldChange: (event: Event, value: {name: string, value: string}) => void
    _onRadioChange: (event: Event, value: {name: string, value: string}) => void

    _onFieldChange(event: Event, value: {name: string, value: string}): void {
        const {value: propsValue, onChange} = this.props;
        const {name: fieldName, value: fieldValue} = value;
        const {mode, fields} = propsValue || {mode: 'bank', fields: BANK_DEFAULTS};

        if (!onChange) {
            return;
        }

        onChange({
            mode,
            fields: assign({}, fields, {[fieldName]: fieldValue}),
        });
    }

    _onRadioChange(event: Event, value: {name: string, value: ModeT}): void {
        const {value: {mode}, onChange} = this.props;
        const {value: nextMode} = value;

        if (!onChange) {
            return;
        }

        if (nextMode !== mode) {
            onChange({
                mode: nextMode,
                fields: nextMode === BANK_MODE ? BANK_DEFAULTS : YANDEX_MONEY_DEFAULTS,
            });
        }
    }

    render(): React.Element<*> | null {
        const {disabled, value: propsValue} = this.props;
        const value = propsValue || {mode: BANK_MODE, fields: BANK_DEFAULTS};

        const {mode} = value;

        const options = [
            {label: __('Bank account'), value: BANK_MODE},
            {label: __('YooMoney'), value: YANDEX_MONEY_MODE},
        ];

        const fields = mode === BANK_MODE
            ? this._renderBankFields()
            : this._renderYandexMoneyFields();

        return (
            <div>
                <Radio
                    className={css.radio}
                    disabled={disabled}
                    name={RADIO_NAME}
                    onChange={this._onRadioChange}
                    options={options}
                    size='l'
                    value={mode}
                />
                {fields}
            </div>
        );
    }

    _renderBankFields(): Array<React.Element<*>> {
        const {disabled, displayError, onBlur, onFocus, value: propsValue} = this.props;
        const {fields} = propsValue || {mode: BANK_MODE, fields: BANK_DEFAULTS};

        // eslint-disable-next-line no-extra-parens
        const {bik, account} = (fields: any);

        const errors = displayError || {};

        const commonProps = {
            className: css.input,
            disabled,
            onBlur,
            onChange: this._onFieldChange,
            onFocus,
        };

        return [
            <FormField
                key='bik'
                displayName={__('BIK')}
                displayError={errors.bik}
            >
                <Input
                    name='bik'
                    value={bik}
                    {...commonProps}
                />
            </FormField>,
            <FormField
                key='account'
                displayName={__('Account')}
                displayError={errors.account}
            >
                <Input
                    name='account'
                    value={account}
                    {...commonProps}
                />
            </FormField>,
        ];
    }

    _renderYandexMoneyFields(): Array<React.Element<*>> {
        const {disabled, displayError, onBlur, onFocus, value: {fields}} = this.props;

        const {
            yandexMoney,
        // eslint-disable-next-line no-extra-parens
        } = (fields: any);

        const errors = displayError || {};

        const commonProps = {
            className: css.input,
            disabled,
            onBlur,
            onChange: this._onFieldChange,
            onFocus,
        };

        return [
            <FormField
                key='yandexMoney'
                displayName={__('YooMoney account')}
                displayError={errors.yandexMoney}
            >
                <Input
                    name='yandexMoney'
                    value={yandexMoney}
                    {...commonProps}
                />
            </FormField>,
        ];
    }
}
