// @flow
'use strict';

const {assign} = Object;

import * as React from 'react';

import {Button, Input, Radio, Select, Link} from 'teatime-components';
import {FormField} from '../FormField/FormField';

import css from './CampaignCreation.css';

type EntryT = {
    bundleId?: string,
    domain?: string,
    mode: 'site' | 'app',
    platform?: string,
    storeLink?: string,
};

type EntryErrorsT = {[fieldId: string]: string};

type PropsT = {
    disabled: boolean,
    displayError: Array<EntryErrorsT>,
    name: string,
    onBlur: () => void,
    onChange: (value: Array<EntryT>) => void,
    onFocus: () => void,
    value: Array<EntryT>,
};

const SITE_MODE = 'site';
const APP_MODE = 'app';

export class CampaignCreation extends React.Component<PropsT> {
    static defaultProps: PropsT = {
        disabled: false,
        displayError: [],
        value: [],
        name: 'field',
        onBlur: () => undefined,
        onChange: () => undefined,
        onFocus: () => undefined,
    }

    constructor(props: PropsT) {
        super(props);

        this._onAddClick = this._onAddClick.bind(this);
        this._onDeleteClick = this._onDeleteClick.bind(this);
        this._onInputChange = this._onInputChange.bind(this);
        this._onRadioChange = this._onRadioChange.bind(this);
    }

    _onAddClick: () => void
    _onDeleteClick: (index: number) => void
    _onInputChange: (value: {name: string, value: string}, index: number) => void
    _onRadioChange: (value: {name: string, value: string}, index: number) => void

    _updateEntry(entries: Array<EntryT>, updatedEntry: EntryT, index: number): Array<EntryT> {
        const nextEntries = entries.slice();
        nextEntries[index] = updatedEntry;
        return nextEntries;
    }

    _onAddClick(): void {
        const {value: entries, onChange} = this.props;

        onChange(entries.concat({mode: SITE_MODE}));
    }

    _onDeleteClick(index: number): void {
        const {value: entries, onChange} = this.props;

        if (index <= 0 || index >= entries.length) {
            return;
        }

        const nextEntries = entries.slice(0, index).concat(entries.slice(index + 1));

        onChange(nextEntries);
    }

    _onInputChange(value: {name: string, value: string}, index: number): void {
        const {value: entries, onChange} = this.props;
        const {name: fieldName, value: nextValue} = value;
        const entry = entries[index];

        if (!entry || !onChange) {
            return;
        }

        const parts = fieldName.split('-');
        const subfieldId = parts[parts.length - 2];

        const nextEntry = assign({}, entry, {[subfieldId]: nextValue});

        onChange(this._updateEntry(entries, nextEntry, index));
    }

    _onRadioChange(value: {name: string, value: 'site' | 'app'}, index: number): void {
        const {value: entries, onChange} = this.props;
        const {value: nextMode} = value;
        const entry = entries[index];

        if (!entry || !onChange) {
            return;
        }

        const {mode} = entry;

        if (nextMode === mode) {
            return;
        }

        const nextEntry = {mode: nextMode};

        onChange(this._updateEntry(entries, nextEntry, index));
    }

    render(): React.Element<*> | null {
        const {displayError: errors, value} = this.props;
        let entries = value;
        let errorEntries = errors || [];

        if (!entries) {
            entries = [
                {mode: SITE_MODE},
            ];
            errorEntries = [
                {},
            ];
        }

        return (
            <div className={css.container}>
                {entries.map((entry, i) => this._renderEntry(entry, errorEntries[i], i))}
                <Button
                    className={css.addButton}
                    theme='normal'
                    size='l'
                    onClick={this._onAddClick}
                >
                    {__('Add more')}
                </Button>
            </div>
        );
    }

    _renderEntry(entry: EntryT, errors: EntryErrorsT, index: number): React.Element<*> {
        const {name} = this.props;
        const {mode} = entry;

        const options = [
            {label: __('Site'), value: SITE_MODE},
            {label: __('Mobile application'), value: APP_MODE},
        ];

        const renderedEntry = mode === SITE_MODE
            ? this._renderSiteEntry(entry, errors, index)
            : this._renderAppEntry(entry, errors, index);

        const deleteButton = index > 0
            ? <div className={css.deleteButton} onClick={() => this._onDeleteClick(index)}/>
            : null;

        return (
            <div className={css.siteContainer} key={`entry-${index}`}>
                <Radio
                    className={css.radio}
                    key={`${name}-mode-${index}`}
                    name={`${name}-mode-${index}`}
                    onChange={(event, value) => this._onRadioChange(value, index)}
                    options={options}
                    size='l'
                    value={mode}
                />
                {deleteButton}
                {renderedEntry}
                <div className={css.hr} />
            </div>
        );
    }

    _renderSiteEntry(entry: Object, errors: Object, index: number): Array<React.Element<*>> {
        const {disabled, name, onBlur, onFocus} = this.props;
        const {domain} = entry || {};
        const {
            domain: domainError,
        } = errors || {};

        const defaultArgs = {
            Component: Input,
            disabled,
            onBlur,
            onChange: (event, value: {value: string, name: string}) => this._onInputChange(value, index),
            onFocus,
        };

        return [
            this._renderField(
                assign({}, defaultArgs, {
                    displayName: __('Domain'),
                    error: domainError,
                    fieldName: `${name}-domain-${index}`,
                    value: domain,
                })
            ),
        ];
    }

    _renderAppEntry(entry: Object, errors: Object, index: number): Array<React.Element<*>> {
        const {disabled, name, onBlur, onFocus} = this.props;
        const {platform, bundleId, storeLink} = entry || {};
        const {
            platform: platformError,
            bundleId: bundleIdError,
            storeLink: storeLinkError,
        } = errors || {};

        const defaultArgs = {
            Component: Input,
            disabled,
            onBlur,
            onChange: (event, value: {value: string, name: string}) => this._onInputChange(value, index),
            onFocus,
        };

        const helpRefBundleIDComponent = (
            <Link href='https://yandex.ru/support/partner2/mobile-apps/app/bundle-id.xml' target='_blank'>
                {__('How to find Bundle ID')}
            </Link>
        );
        return [
            this._renderField(
                assign({}, defaultArgs, {
                    Component: Select,
                    displayName: __('Platform'),
                    error: platformError,
                    fieldName: `${name}-platform-${index}`,
                    value: platform,
                    options: [
                        {label: __('Android'), value: 'android'},
                        {label: __('iOS'), value: 'ios'},
                    ],
                })
            ),

            this._renderField(
                assign({}, defaultArgs, {
                    displayName: __('BundleID'),
                    error: bundleIdError,
                    fieldName: `${name}-bundleId-${index}`,
                    value: bundleId,
                    placeholder: 'com.yandex.browser',
                    displaySubHint: helpRefBundleIDComponent,
                })
            ),

            this._renderField(
                assign({}, defaultArgs, {
                    displayName: __('Application link'),
                    error: storeLinkError,
                    fieldName: `${name}-storeLink-${index}`,
                    value: storeLink,
                })
            ),
        ];
    }

    _renderField(args: Object): React.Element<*> {
        const {
            Component,
            className,
            componentClassName,
            disabled,
            displayName,
            displaySubHint,
            error,
            fieldName,
            onBlur,
            onChange,
            onFocus,
            options,
            type,
            value,
            placeholder,
        } = args;

        return (
            <FormField
                className={className}
                componentClassName={componentClassName}
                key={`campaign-${fieldName}`}
                displayName={displayName}
                displayError={error}
                displaySubHint={displaySubHint}
            >
                <Component
                    className={css.input}
                    disabled={disabled}
                    name={fieldName}
                    onBlur={onBlur}
                    onChange={onChange}
                    onFocus={onFocus}
                    options={options}
                    size='l'
                    type={type}
                    value={value}
                    placeholder={placeholder}
                />
            </FormField>
        );
    }
}
