// @flow
'use strict';

import * as React from 'react';
import moment from 'moment';
import DayPicker from 'react-day-picker/DayPicker';
import DayPickerInput from 'react-day-picker/DayPickerInput';
import classnames from 'classnames';
import {Input, Select} from 'teatime-components';

import MomentLocaleUtils, {
    formatDate,
    parseDate,
} from 'react-day-picker/moment';

import 'moment/locale/ru';

import 'react-day-picker/lib/style.css';

const classesDayPicker = DayPicker.defaultProps.classNames;

import css from './DatePicker.css';

const currentYear = new Date().getFullYear();
const fromMonth = new Date(currentYear - 100, 0);
const toMonth = new Date(currentYear, 11);

type YearMonthSelectPropsT = {
    date: Date,
    locale: string,
    localeUtils: Object,
    onChange: (value: Date) => void,
}

class YearMonthSelect extends React.Component<YearMonthSelectPropsT> {
    constructor(props: YearMonthSelectPropsT) {
        super(props);

        this._onChange = this._onChange.bind(this);
    }

    _onChange: (event: Event) => void

    _onChange(event: Event, {name, value}) {
        const {date, onChange} = this.props;

        if (name === 'month') {
            onChange(new Date(date.getFullYear(), value));
        } else if (name === 'year') {
            onChange(new Date(value, date.getMonth()));
        }
    }

    render(): React.Element<*> {
        const {date, locale, localeUtils} = this.props;

        const months = localeUtils.getMonths(locale);

        const years = [];
        for (let i = fromMonth.getFullYear(); i <= toMonth.getFullYear(); i++) {
            years.push(i);
        }

        const monthOptions = months.map((month, i) => ({label: month, value: i.toString()}));
        const yearOptions = years.map(year => ({label: year, value: year.toString()}));

        return (
            <form className={css.caption}>
                <Select
                    className={css.monthSelect}
                    name='month'
                    onChange={this._onChange}
                    options={monthOptions}
                    size='s'
                    value={date.getMonth().toString()}
                />
                <Select
                    className={css.yearSelect}
                    name='year'
                    onChange={this._onChange}
                    options={yearOptions}
                    size='s'
                    value={date.getFullYear().toString()}
                />
            </form>
        );
    }
}

type PropsT = {
    date: Date | null,
    disabled: boolean,
    language: string,
    olderThan18: boolean,
    onBlur: () => void,
    onChange: (date: string) => void,
    onFocus?: () => void,
}

type StateT = {
    month: Date | null,
}

export class DatePicker extends React.Component<PropsT, StateT> {
    constructor(props: PropsT) {
        super(props);

        this._onChange = this._onChange.bind(this);
        this._onYearMonthChange = this._onYearMonthChange.bind(this);

        this.state = {month: null};
    }

    /*
    * Хак против необходимости кликать дважды на селекты в
    * календаре. Из пропсов приходит ошибка, когда значеение
    * например равно начальному (null)
    */
    shouldComponentUpdate(nextProps: PropsT, nextState: StateT) {
        return this.state.month !== nextState.month;
    }

    _onChange: (date: string) => void
    _onYearMonthChange: (month: Date) => void

    _onChange(date: string) {
        const {onChange, onBlur} = this.props;

        onChange(date);
        /*
        * после onChange окно datepicker пропадает и необходимо
        * вызвать валидацию поля поскольку значение изменилось
        * onBlue делает проверку валидации. Как только появится
        * валидации на onChange можно попробовать убрать этот код
        */
        onBlur();
    }

    _onYearMonthChange(month: Date) {
        this.setState({month});
    }

    render() {
        const {date, disabled, language, olderThan18, onBlur, onFocus} = this.props;
        const {month} = this.state;

        const classNames = {
            container: classnames(css.container, 'DayPickerInput'),
            overlayWrapper: 'DayPickerInput-OverlayWrapper',
            overlay: 'DayPickerInput-Overlay',
        };

        const disabledDays = olderThan18
            ? {after: moment().subtract(18, 'years').toDate()}
            : undefined;

        const dayPickerProps = {
            locale: language,
            localeUtils: MomentLocaleUtils,
            selectedDays: date,
            classNames: Object.assign({}, classesDayPicker, {
                month: `${classesDayPicker.month} ${css.month}`,
                disabled: 'DayPicker-Day--disabled',
            }),
            disabledDays,
            month,
            captionElement: ({date, localeUtils}) => (
                <YearMonthSelect
                    date={date}
                    locale={language}
                    localeUtils={localeUtils}
                    onChange={this._onYearMonthChange}
                />
            ),
        };

        const inputProps = {
            disabled,
            name: 'date-input',
            className: css.removeClear,
            size: 'l',
            onFocus,
            onBlur,
        };

        return (
            <div>
                <DayPickerInput
                    component={Input}
                    classNames={classNames}
                    dayPickerProps={dayPickerProps}
                    formatDate={formatDate}
                    keepFocus={false}
                    parseDate={parseDate}
                    placeholder={`${formatDate(new Date(), 'L', language)}`}
                    inputProps={inputProps}
                    onDayChange={this._onChange}
                />
            </div>
        );
    }
}
