// @flow
'use strict';

import * as React from 'react';
import {TransitionMotion, spring} from 'react-motion';

import css from './ErrorStack.css';

// @ see https://github.com/chenglou/react-motion#helpers
const SPRING_PRESET = {stiffness: 300, damping: 15};

type ErrorWithIdT = {
    id: string,
    error: Error,
};

type ErrorTransitionDataT = {
    data?: ErrorWithIdT,
    key: string,
    style: {
        top: number,
        opacity: number,
    }
}

type ErrorTransitionDataDeclarationT = {
    data?: ErrorWithIdT,
    key: string,
    style: {
        top: *, // result of spring(...)
        opacity: *, // result of spring(...)
    }
}

type PropsT = {
    errors: Array<ErrorWithIdT>,
    onDeleteClick: (id: string) => void,
};

export class ErrorStack extends React.Component<PropsT> {
    static defaultProps = {
        onDeleteClick: () => {},
    }

    constructor(props: PropsT) {
        super(props);

        this._getTransitionData = this._getTransitionData.bind(this);
    }

    render() {
        return (
            <div className={css.root}>
                <TransitionMotion
                    willEnter={this._willEnter}
                    willLeave={this._willLeave}
                    styles={this._getTransitionData}>
                    {errors => this._renderErrors(errors)}
                </TransitionMotion>
            </div>
        );
    }

    _willEnter() {
        return {top: -40, opacity: 0};
    }

    _willLeave() {
        return {top: 0, opacity: 1};
    }

    _renderErrors(errors: Array<ErrorTransitionDataT>): React.Element<*> {
        return (
            <div>
                {errors.map(error => this._renderError(error))}
            </div>
        );
    }

    /*
     * onMouseDown здесь важно, чтобы перехватить другие обработчики,
     * реализованные через react-outside-event
     */
    _renderError({data, style, key}: ErrorTransitionDataT): React.Element<*> | null {
        if (!data) {
            return null;
        }

        const {error: {message}} = data;

        return (
            <div key={key} className={css.error} style={style}>
                <div
                    className={css.x}
                    onMouseDown={(event: Event) => this._onDeleteClick(event, key)}>
                    &times;
                </div>
                <div className={css.message}>
                    {message}
                </div>
            </div>
        );
    }

    _getTransitionData: () => Array<ErrorTransitionDataDeclarationT>

    _getTransitionData(): Array<ErrorTransitionDataDeclarationT> {
        const {errors} = this.props;

        return errors.map(error => {
            const style = {
                top: spring(0, SPRING_PRESET),
                opacity: spring(1, SPRING_PRESET),
            };
            return {style, data: error, key: error.id};
        });
    }

    _onDeleteClick(event: Event, id: string) {
        event.stopPropagation();
        this.props.onDeleteClick(id);
    }
}
