// @flow
'use strict';

import * as React from 'react';

import {Button, Spin} from 'teatime-components';

import css from './FileUpload.css';

import type {FileDescT} from '../../types/state';

type PropsT = {
    disabled: boolean,
    files: Array<FileDescT>,
    name: string,
    onChange: (file: File | null) => void,
    onDelete: (fileId: string) => void,
}

export class FileUpload extends React.Component<PropsT> {
    fileInput: HTMLInputElement | null

    constructor(props: PropsT) {
        super(props);

        this._onChange = this._onChange.bind(this);
        this._onButtonClick = this._onButtonClick.bind(this);
    }

    _onChange: (event: SyntheticInputEvent<HTMLInputElement>) => void
    _onButtonClick: (event: SyntheticMouseEvent<HTMLButtonElement>) => void

    _onButtonClick() {
        if (this.fileInput) {
            this.fileInput.click();
        }
    }

    _onChange() {
        const {onChange} = this.props;

        if (!this.fileInput) {
            return;
        }

        const {files} = this.fileInput;
        const file = files[0] || null;

        onChange(file);
    }

    render(): React.Element<*> {
        return (
            <div className={css.container}>
                {this._renderButton()}
                {this._renderFiles()}
            </div>
        );
    }

    _renderButton(): React.Element<*> {
        const {disabled, name} = this.props;

        return (
            <Button
                className={css.button}
                disabled={disabled}
                size='l'
                theme='normal'
                onClick={this._onButtonClick}
            >
                {__('Upload')}
                <input
                    type='file'
                    onChange={this._onChange}
                    name={name}
                    ref={input => {
                        this.fileInput = input;
                    }}
                />
            </Button>
        );
    }

    _renderFiles(): React.Element<*> {
        const {files} = this.props;

        if (files.length <= 0) {
            return (
                <div className={css.fileListContainer}>
                    <div className={css.fileContainer}>
                        <div className={css.filename}>
                            {__('Choose a file')}
                        </div>
                    </div>
                </div>
            );
        }

        return (
            <div className={css.fileListContainer}>
                {files.map(file => this._renderFile(file))}
            </div>
        );
    }

    _renderFile(fileDesc: FileDescT): React.Element<*> {
        const {
            file: {name: filename},
            id,
            ok,
            pending,
        } = fileDesc;

        const {onDelete} = this.props;

        const spin = pending
            ? <Spin size='xs' />
            : null;

        const check = ok
            ? <div className={css.check} />
            : null;

        return (
            <div className={css.fileContainer} key={id}>
                <div className={css.filename}>
                    {filename}
                </div>
                <div className={css.controls}>
                    {spin}
                    {check}
                    <div
                        className={css.cross}
                        onClick={() => onDelete(id)}
                    />
                </div>
            </div>
        );
    }
}
