// @flow
'use strict';

const {assign} = Object;

import {omit} from 'lodash';
import * as React from 'react';
import classnames from 'classnames';

import {HintBox} from '../HintBox/HintBox';

import css from './FormField.css';

type PropsT = {
    Component?: React.ComponentType<*>,
    children?: *,
    className?: Object,
    componentClassName?: Object,
    displayError?: string | Object,
    displaySubHint?: string | React.Element<*>,
    displayName?: string | null,
    focused?: boolean,
    hint?: string | Object,
    isScrollTarget?: boolean,
    value?: *,
    visible?: boolean,
};

export class FormField extends React.Component<PropsT> {
    _elementRef: Element | null;

    componentDidMount(): void {
        const {isScrollTarget} = this.props;

        if (isScrollTarget) {
            this._scrollIntoView();
        }
    }

    componentDidUpdate(prevProps: PropsT): void {
        const {isScrollTarget} = this.props;
        const {isScrollTarget: prevIsScrollTarget} = prevProps;

        if (!prevIsScrollTarget && isScrollTarget) {
            this._scrollIntoView();
        }
    }

    render(): React.Node {
        const {Component, children, className, componentClassName, visible = true} = this.props;

        if (!visible) {
            return null;
        }

        let element = null;

        const childrenProps = assign(
            omit(this.props, ['Component', 'children', 'displayName']),
            {className: classnames(css.control, componentClassName)},
        );

        if (children) {
            element = React.cloneElement(children, childrenProps);
        } else if (Component) {
            element = <Component {...childrenProps} />;
        }

        return (
            <div
                className={classnames(css.fieldContainer, className)}
                ref={ref => { this._elementRef = ref; }}
            >
                {this._renderTitle()}
                {this._renderHint()}
                <div className={css.field}>
                    {element}
                </div>
                {this._renderError()}
                {this._renderSubHint()}
            </div>
        );
    }

    _renderTitle(): React.Element<*> | null {
        const {displayName} = this.props;

        if (!displayName) {
            return null;
        }

        return (
            <div className={css.fieldTitle}>
                {displayName}
            </div>
        );
    }

    _renderSubHint(): React.Element<*> | null {
        const {displaySubHint, hint, displayError} = this.props;

        // displaySubHint - показываем всегда (используется у поля CampaignCreation)
        // hint показываем если есть ошибка
        const finalSubHint = displaySubHint || displayError && hint;

        if (!finalSubHint) {
            return null;
        }
        const classes = typeof finalSubHint === 'string' ? css.subHint : null;
        return (
            <div className={classes}>
                {finalSubHint}
            </div>
        );
    }

    _renderError(): React.Element<*> | null {
        const {displayError} = this.props;

        if (!displayError || typeof displayError === 'object') {
            return null;
        }

        return (
            <div className={css.error}>
                {displayError}
            </div>
        );
    }

    _renderHint(): React.Element<*> | null {
        const {hint, displayError, focused} = this.props;

        // если есть ошибка или нет фокуса на поле, то подсказку справа не показываем
        if (displayError || !focused || !hint || typeof hint === 'object') {
            return null;
        }

        return (
            <HintBox>
                {hint}
            </HintBox>
        );
    }

    _scrollIntoView(): void {
        const ref = this._elementRef;

        if (ref) {
            ref.scrollIntoView();
        }
    }
}
