// @flow
'use strict';

import * as React from 'react';

import {Input, Radio} from 'teatime-components';

import css from './IbanOrAccount.css';

type PropsT = {
    disabled: boolean,
    value: {
        mode: 'iban' | 'account',
        value: string,
    },
    onBlur?: () => void,
    onChange?: (value: {mode: string, value: string}) => void,
    onFocus?: () => void,
};

const RADIO_NAME = 'ibanOrAccountRadio';
const INPUT_NAME = 'ibanOrAccountInput';

export class IbanOrAccount extends React.Component<PropsT> {
    static defaultProps: PropsT = {
        disabled: false,
        value: {
            mode: 'iban',
            value: '',
        },
    }

    constructor(props: PropsT) {
        super(props);

        this._onInputChange = this._onInputChange.bind(this);
        this._onRadioChange = this._onRadioChange.bind(this);
    }

    _onInputChange: (event: Event, value: {name: string, value: string}) => void
    _onRadioChange: (event: Event, value: {name: string, value: string}) => void

    _onInputChange(event: Event, value: {name: string, value: string}): void {
        const {value: {mode}, onChange} = this.props;
        const {value: nextValue} = value;

        if (!onChange) {
            return;
        }

        onChange({mode, value: nextValue});
    }

    _onRadioChange(event: Event, value: {name: string, value: string}): void {
        const {value: {mode}, onChange} = this.props;
        const {value: nextMode} = value;

        if (!onChange) {
            return;
        }

        if (nextMode !== mode) {
            onChange({mode: nextMode, value: ''});
        }
    }

    render(): React.Element<*> {
        const {disabled, onBlur, onFocus, value: {mode, value}} = this.props;

        const options = [
            {label: __('IBAN'), value: 'iban'},
            {label: __('Account'), value: 'account'},
        ];

        return (
            <div>
                <Radio
                    className={css.radio}
                    disabled={disabled}
                    name={RADIO_NAME}
                    onChange={this._onRadioChange}
                    options={options}
                    size='l'
                    value={mode}
                />
                <Input
                    className={css.input}
                    disabled={disabled}
                    name={INPUT_NAME}
                    onBlur={onBlur}
                    onChange={this._onInputChange}
                    onFocus={onFocus}
                    value={value}
                />
            </div>
        );
    }
}
