// @flow
'use strict';

const {Component} = require('react');
const {noop} = require('lodash');
const Overlay = require('../Overlay/Overlay');
const React = require('react');
const ReactOutsideEvent = require('react-outside-event');
const classNames = require('classnames');

const css = require('./LangSwitcher.css');

const locales = {
    en: 'English',
    ru: 'Русский',
};

type LangsT = 'en' | 'ru';

type Props = {
    availableLocales: Array<LangsT>,
    className?: string,
    locale: LangsT,
    onChange: (e: SyntheticEvent<>, selectedLocale: LangsT) => void,
};

type State = {
    isOpened: boolean,
};

class LangSwitcher extends Component<Props, State> {
    static defaultProps = {
        availableLocales: ['en', 'ru'],
        locale: 'ru',
        onChange: noop,
    }

    constructor(props: Props) {
        super(props);

        this.state = {
            isOpened: false,
        };

        this._toggleMenu = this._toggleMenu.bind(this);
    }

    _onSelect(e: SyntheticEvent<>, selectedLocale: LangsT) {
        this.props.onChange(e, selectedLocale);
    }

    _toggleMenu: () => void

    _toggleMenu() {
        this.setState({isOpened: !this.state.isOpened});
    }

    onOutsideEvent() {
        if (this.state.isOpened) {
            this.setState({isOpened: false});
        }
    }

    render() {
        const {className} = this.props;

        return (
            <div className={classNames(css.wrapper, className)} >
                {this.renderLabel()}
                {this.renderMenu()}
            </div>
        );
    }

    renderLabel(): React.Element<*> {
        const {locale} = this.props;

        return (
            <span className={css.label} onClick={this._toggleMenu}>
                <i className={css[locale]}/>
                {locales[locale]}
            </span>
        );
    }

    renderMenu(): React.Element<*> | null {
        const {isOpened} = this.state;

        if (!isOpened) {
            return null;
        }

        const {availableLocales, locale} = this.props;
        const localesView = availableLocales.map(availableLocale =>
            this.renderMenuItem(availableLocale, locale));

        return (
            <Overlay shown strategy={['bottom']}>
                {localesView}
            </Overlay>
        );
    }

    renderMenuItem(locale: LangsT, selectedLocale: LangsT): React.Element<*> {
        const isSelectedLang = locale === selectedLocale;
        const clickHandler = isSelectedLang ? null : e => this._onSelect(e, locale);

        return (
            <span
                className={classNames(css.menuItem, {
                    [css.menuActiveItem]: isSelectedLang,
                })}
                key={`_${locale}`}
                onClick={clickHandler}>
                <i className={css[locale]}/>
                {locales[locale]}
            </span>
        );
    }
}

module.exports = ReactOutsideEvent(LangSwitcher);
