// @flow
'use strict';

const {assign} = Object;
import * as React from 'react';

type PropsT = {
    onChange: (date: string) => void,
}

type OptionsT = {
    mask: string,
    maskChar: string,
}

type MarkerT = {
    pos: number,
    char: string,
}

export function withMask(WrappedComponent: React.ComponentType<*>, {mask, maskChar}: OptionsT) {
    const markers = createMarkers(mask, maskChar);

    return class extends React.Component<PropsT> {
        constructor(props: PropsT) {
            super(props);

            this._onChange = this._onChange.bind(this);
        }

        _onChange: (event: Event, value: {name: string, value: string}) => void

        _onChange(event: Event, payload: {name: string, value: string}) {
            const {onChange} = this.props;
            const {value} = payload;

            onChange(applyMarkers(value, markers));
        }

        render(): React.Element<*> {
            const componentProps = assign({}, this.props, {onChange: this._onChange});

            return <WrappedComponent {...componentProps} />;
        }
    };
}

function createMarkers(mask: string = 'XXX-XXX', maskChar: string = 'X'): Array<MarkerT> {
    return mask.split('').reduce((acc, char, i) => {
        if (char !== maskChar) {
            acc.push({pos: i, char});
        }

        return acc;
    }, []);
}

function applyMarkers(value: string, markers: Array<MarkerT>): string {
    const result = value.split('');

    for (const marker of markers) {
        const {pos, char} = marker;

        if (pos >= result.length) {
            break;
        }

        if (result[pos] !== char) {
            result.splice(pos, 0, char);
        }
    }

    return result.join('');
}
