// @flow
'use strict';

const {assign} = Object;

import * as React from 'react';

import {Button, Input, Select} from 'teatime-components';
import {FormField} from '../FormField/FormField';

import css from './MobileAppCreation.css';

import type {MobileAppCreationEntryT as EntryT} from '../../types/state';

type EntryErrorsT = {[fieldId: string]: string};

type PropsT = {
    disabled: boolean,
    displayError: Array<EntryErrorsT>,
    name: string,
    onBlur: () => void,
    onChange: (value: Array<EntryT>) => void,
    onFocus: () => void,
    value: Array<EntryT>,
};

export class MobileAppCreation extends React.Component<PropsT> {
    static defaultProps: PropsT = {
        disabled: false,
        displayError: [],
        value: [],
        name: 'field',
        onBlur: () => undefined,
        onChange: () => undefined,
        onFocus: () => undefined,
    }

    constructor(props: PropsT) {
        super(props);

        this._onAddClick = this._onAddClick.bind(this);
        this._onDeleteClick = this._onDeleteClick.bind(this);
        this._onInputChange = this._onInputChange.bind(this);
    }

    _onAddClick: () => void
    _onDeleteClick: (index: number) => void
    _onInputChange: (value: {name: string, value: string}, index: number) => void

    _updateEntry(entries: Array<EntryT>, updatedEntry: EntryT, index: number): Array<EntryT> {
        const nextEntries = entries.slice();
        nextEntries[index] = updatedEntry;
        return nextEntries;
    }

    _onAddClick(): void {
        const {value: entries, onChange} = this.props;

        onChange(entries.concat({
            bundleId: '',
            platform: '',
            storeLink: '',
        }));
    }

    _onDeleteClick(index: number): void {
        const {value: entries, onChange} = this.props;

        if (index <= 0 || index >= entries.length) {
            return;
        }

        const nextEntries = entries.slice(0, index).concat(entries.slice(index + 1));

        onChange(nextEntries);
    }

    _onInputChange(value: {name: string, value: string}, index: number): void {
        const {value: entries, onChange} = this.props;
        const {name: fieldName, value: nextValue} = value;
        const entry = entries[index];

        if (!entry || !onChange) {
            return;
        }

        const parts = fieldName.split('-');
        const subfieldId = parts[parts.length - 2];

        const nextEntry = assign({}, entry, {[subfieldId]: nextValue});

        onChange(this._updateEntry(entries, nextEntry, index));
    }

    render(): React.Element<*> | null {
        const {displayError: errors, value} = this.props;
        let entries = value;
        let errorEntries = errors || [];

        if (!entries) {
            entries = [
                {
                    bundleId: '',
                    platform: '',
                    storeLink: '',
                },
            ];
            errorEntries = [
                {},
            ];
        }

        return (
            <div className={css.container}>
                {entries.map((entry, i) => this._renderEntry(entry, errorEntries[i], i))}
                <Button
                    className={css.addButton}
                    theme='normal'
                    size='l'
                    onClick={this._onAddClick}
                >
                    {__('Add more')}
                </Button>
            </div>
        );
    }

    _renderEntry(entry: EntryT, errors: EntryErrorsT, index: number): React.Element<*> {
        const renderedEntry = this._renderAppEntry(entry, errors, index);

        const deleteButton = index > 0
            ? <Button className={css.deleteButton} size='l' theme='link' onClick={() => this._onDeleteClick(index)}>{__('Delete')}</Button>
            : null;

        return (
            <div className={css.siteContainer} key={`entry-${index}`}>
                {deleteButton}
                {renderedEntry}
                <div className={css.hr} />
            </div>
        );
    }

    _renderAppEntry(entry: Object, errors: Object, index: number): Array<React.Element<*>> {
        const {disabled, name, onBlur, onFocus} = this.props;
        const {platform, bundleId, storeLink} = entry || {};
        const {
            platform: platformError,
            bundleId: bundleIdError,
            storeLink: storeLinkError,
        } = errors || {};

        const defaultArgs = {
            Component: Input,
            disabled,
            onBlur,
            onChange: (event, value: {value: string, name: string}) => this._onInputChange(value, index),
            onFocus,
        };

        return [
            this._renderField(
                assign({}, defaultArgs, {
                    Component: Select,
                    displayName: __('Platform'),
                    error: platformError,
                    fieldName: `${name}-platform-${index}`,
                    value: platform,
                    options: [
                        {label: __('Android'), value: 'android'},
                        {label: __('iOS'), value: 'ios'},
                    ],
                })
            ),

            this._renderField(
                assign({}, defaultArgs, {
                    displayName: __('BundleID'),
                    error: bundleIdError,
                    fieldName: `${name}-bundleId-${index}`,
                    value: bundleId,
                })
            ),

            this._renderField(
                assign({}, defaultArgs, {
                    displayName: __('Application link'),
                    error: storeLinkError,
                    fieldName: `${name}-storeLink-${index}`,
                    value: storeLink,
                })
            ),
        ];
    }

    _renderField(args: Object): React.Element<*> {
        const {
            Component,
            className,
            componentClassName,
            disabled,
            displayName,
            error,
            fieldName,
            onBlur,
            onChange,
            onFocus,
            options,
            type,
            value,
        } = args;

        return (
            <FormField
                className={className}
                componentClassName={componentClassName}
                key={`mobileApp-${fieldName}`}
                displayName={displayName}
                displayError={error}
            >
                <Component
                    className={css.input}
                    disabled={disabled}
                    name={fieldName}
                    onBlur={onBlur}
                    onChange={onChange}
                    onFocus={onFocus}
                    options={options}
                    size='l'
                    type={type}
                    value={value}
                />
            </FormField>
        );
    }
}
