// @flow
'use strict';

import * as React from 'react';
import {Button, Spin} from 'teatime-components';
import Markdown from 'markdown-to-jsx';
import classnames from 'classnames';

import {Logo, User} from '../../../packages/partner-components';
import {HeadContainer} from '../../simple/containers/HeadContainer.tsx';
import LangSwitcher from '../LangSwitcher/LangSwitcher';

import {shouldRedirectUser} from '../../lib/redirects';

import css from './Page.css';

import type {UserDataT} from '../../types/state';

const CANONICAL_FORM_URL = 'https://partner2.yandex.ru/form/';

type ComponentT = Class<React$Component<*, *>>;

type PropsT = {
    ContentContainer: ComponentT,
    ErrorStackContainer: ComponentT,
    children?: React.Node,
    onChangeLang: (lang: string) => void,
    onSubmit: () => void,
    pending?: boolean,
    submitDisabled: boolean,
    user?: UserDataT,
    location: {
        pathname: string,
        search: string,
        hash: string,
        state: Object,
    },
    isSimpleForm: boolean,
}

export class Page extends React.Component<PropsT> {
    static defaultProps = {
        ContentContainer: null,
    }

    render(): React.Element<*> | null {
        const {
            user,
            location: {
                pathname,
            },
            isSimpleForm,
        } = this.props;

        if (
            pathname.startsWith('/v2/dashboard') ||
            pathname.startsWith('/v2/mobile-mediation')
        ) {
            return this._renderDashboardStub();
        }

        if (!user || !user.isLogged) {
            return null;
        }

        const {
            canFillForm,
            canFillFormMessage,
            hasRoleInPartner2,
        } = user;

        if (!canFillForm) {
            if (shouldRedirectUser(user)) {
                return null;
            }

            if (!hasRoleInPartner2) {
                return this._renderForbidden(canFillFormMessage);
            }
        }

        const containerTypeStyle = isSimpleForm ? css.container_simple : css.container_old;

        return (
            <div className={classnames(css.container, containerTypeStyle)}>
                {this._renderHeader()}
                {this._renderErrorStack()}
                {this._renderContent()}
                {this._renderFooter()}
            </div>
        );
    }

    _renderForbidden(message?: string): React.Element<*> {
        const text = message
            ? replaceBacklinks(message)
            : __('You cannot fill form');
        return (
            <div className={css.forbiddenParent}>
                <div className={css.forbiddenChild}>
                    <h1>
                        <TextWithLink
                            text={text}
                        />
                    </h1>
                </div>
            </div>
        );
    }

    _renderDashboardStub(): React.Element<*> {
        return (
            <div className={css.forbiddenParent}>
                <div className={css.forbiddenChild}>
                    <h1>{__('This is a dashboard stub')}</h1>
                </div>
            </div>
        );
    }

    _renderHeader(): React.Element<*> | null {
        const {user = {}, isSimpleForm, onChangeLang} = this.props;
        const {userId, displayLogin, language} = user;

        if (isSimpleForm) {
            return (
                <HeadContainer/>
            );
        }

        return (
            <>
                <div className={css.header}>
                    <div>
                        <Logo className={css.logo} locale={language} />
                    </div>
                    <div>
                        <LangSwitcher className={css.langSwitcher} locale={language} onChange={onChangeLang} />
                        <User avatarId={userId} menuItems={[]}>{displayLogin}</User>
                    </div>
                </div>
                <div className={css.headerMargin} />
            </>
        );
    }

    _renderErrorStack(): React.Element<*> {
        const {ErrorStackContainer} = this.props;

        return <ErrorStackContainer />;
    }

    _renderContent(): React.Element<*> | null {
        const {children} = this.props;
        const {ContentContainer} = this.props;

        const childrenArray = React.Children.toArray(children);
        const child = childrenArray[0];

        if (child) {
            return child;
        }

        if (!ContentContainer) {
            return null;
        }

        return <ContentContainer />;
    }

    _renderFooter(): React.Element<*> | null {
        const {submitDisabled: disabled, onSubmit, pending, isSimpleForm} = this.props;
        const {user = {}} = this.props;
        const {force} = user;

        if (isSimpleForm) {
            return null;
        }

        const label = force
            ? __('Send')
            : __('Send application');

        const spin = pending
            ? <Spin className={css.spin} size='m' />
            : null;

        return (
            <div className={css.footer}>
                <Button
                    disabled={disabled}
                    onClick={onSubmit}
                    size='l'
                    theme='action'
                    id='submitFormButton'
                >
                    {label}
                </Button>
                {spin}
            </div>
        );
    }
}

function TextWithLink({text}: {text: string}): React.Element<*> {
    const markdown = text.replace(/<(http.*?)>/gi, '[$1]($1)');

    return (
        <Markdown>
            {markdown}
        </Markdown>
    );
}

/**
 * Скорректировать URL анкеты в параметре возврата из паспорта и в прямой ссылке.
 * Бэкенд присылает текст сообщения с ссылками на "канонический" URL анкеты,
 * реальная же ссылка будет отличаться во-первых хостом в разных окружениях,
 * во-вторых префиксом пути для разных веток анкеты (например, /form/assessor/).
 * Фронт заменяет канонический URL на текущий.
 *
 * @param {string} text
 * @return {string}
 */
function replaceBacklinks(text: string): string {
    const currentUrl = location.href;
    const retpath = encodeURIComponent(currentUrl);
    const formUrlRx = new RegExp(CANONICAL_FORM_URL.replace(/[.]/g, '\\.'), 'gi');
    return text.replace(formUrlRx, currentUrl)
        .replace(/\bretpath=[\w%.,+-]*\b/gi, `retpath=${retpath}`);
}
