// @flow
'use strict';

const {assign} = Object;
import * as React from 'react';

import {Input} from 'teatime-components';
import {FormField} from '../FormField/FormField';

import type {YandexMoneyDetailsT as ValueT} from '../../types/state';

import css from './YandexMoneyDetails.css';

const YANDEX_MONEY_MODE = 'yandexMoney';

type PropsT = {
    disabled: boolean,
    displayError: Object,
    value: ValueT,
    hint: Object,
    onBlur?: () => void,
    onChange?: (value: ValueT) => void,
    onFocus?: () => void,
};

const YANDEX_MONEY_DEFAULTS = {
    firstName: '',
    lastName: '',
    passportNumber: '',
    patronymicName: '',
    yandexMoney: '',
};

export class YandexMoneyDetails extends React.Component<PropsT> {
    static defaultProps: PropsT = {
        displayError: {},
        disabled: false,
        hint: {},
        value: {
            mode: YANDEX_MONEY_MODE,
            fields: YANDEX_MONEY_DEFAULTS,
        },
    }

    _onFieldChange = (event: Event, value: {name: string, value: string}): void => {
        // eslint-disable-next-line no-invalid-this
        const {value: propsValue, onChange} = this.props;
        const {name: fieldName, value: fieldValue} = value;
        const {mode, fields} = propsValue || {mode: YANDEX_MONEY_MODE, fields: YANDEX_MONEY_DEFAULTS};

        if (!onChange) {
            return;
        }

        onChange({
            mode,
            fields: assign({}, fields, {[fieldName]: fieldValue}),
        });
    }

    render(): React.Element<*> | null {
        return (
            <div>
                {this._renderYandexMoneyFields()}
            </div>
        );
    }

    _renderYandexMoneyFields(): Array<React.Element<*>> {
        const {disabled, displayError, onBlur, onFocus, value: propsValue} = this.props;
        const {fields} = propsValue || {mode: YANDEX_MONEY_MODE, fields: YANDEX_MONEY_DEFAULTS};

        const {
            firstName,
            lastName,
            passportNumber,
            patronymicName,
            yandexMoney,
        // eslint-disable-next-line no-extra-parens
        } = (fields: any);

        const errors = displayError || {};

        const commonProps = {
            className: css.input,
            disabled,
            onBlur,
            onChange: this._onFieldChange,
            onFocus,
        };

        return [
            <FormField
                key='passportNumber'
                displayName={__('Passport number')}
                displayError={errors.passportNumber}
            >
                <Input
                    name='passportNumber'
                    value={passportNumber}
                    {...commonProps}
                />
            </FormField>,
            <FormField
                key='firstName'
                displayName={__('First name')}
                displayError={errors.firstName}
            >
                <Input
                    name='firstName'
                    value={firstName}
                    {...commonProps}
                />
            </FormField>,
            <FormField
                key='lastName'
                displayName={__('Last name')}
                displayError={errors.lastName}
            >
                <Input
                    name='lastName'
                    value={lastName}
                    {...commonProps}
                />
            </FormField>,
            <FormField
                key='patronymicName'
                displayName={__('Patronymic name')}
                displayError={errors.patronymicName}
            >
                <Input
                    name='patronymicName'
                    value={patronymicName}
                    {...commonProps}
                />
            </FormField>,
            <FormField
                key='yandexMoney'
                displayName={__('YooMoney account')}
                displayError={errors.yandexMoney}
            >
                <Input
                    key='yandexMoney'
                    name='yandexMoney'
                    value={yandexMoney}
                    {...commonProps}
                />
            </FormField>,
        ];
    }
}
