// @flow
'use strict';

import * as React from 'react';
import {connect} from 'react-redux';
import {Check, Input, Select, Textarea, Radio} from 'teatime-components';

const {assign} = Object;

import {getUniqId} from '../../lib/utils';

import {
    fieldBlur,
    fieldChange,
    fieldFocus,
    fileUploadFieldAdd,
    fileUploadFieldDelete,
    fetchFias,
} from '../../actions';

import {FormField} from '../../components/FormField/FormField';

import {DatePicker} from '../../components/DatePicker/DatePicker';
import {FileUpload} from '../../components/FileUpload/FileUpload';
import {IbanOrAccount} from '../../components/IbanOrAccount/IbanOrAccount';
import {OfferCheck} from '../../components/OfferCheck/OfferCheck';
import {Address} from '../../components/Address/Address';
import {Text} from '../../components/Text/Text';
import {GfmText} from '../../components/GfmText/GfmText';
import {YandexMoneyDetails} from '../../components/YandexMoneyDetails/YandexMoneyDetails';
import {BikWithAccountOrYandexMoney} from '../../components/BikWithAccountOrYandexMoney/BikWithAccountOrYandexMoney';
import {CampaignCreation} from '../../components/CampaignCreation/CampaignCreation';
import {MobileAppCreation} from '../../components/MobileAppCreation/MobileAppCreation';
import {AdfoxAccount} from '../../components/AdfoxAccount/AdfoxAccount';

import {withMask} from '../../components/MaskHOC/MaskHOC';

import {selectStoreFieldPending} from '../../selectors/fields';

import type {Dispatch} from 'redux';

import type {FieldDefT} from '../../types/fields';
import type {ApplicationStateT} from '../../types/state';

type StatePropsT = {
    name?: string,
    value?: string,
    error?: string | null,
};

type DispatchPropsT = {
    onBlur: () => *,
    onChange: Function,
    onFocus: () => *,
};

function _baseMapStateToProps(
    state: ApplicationStateT,
    fieldDef: FieldDefT,
    Component: React.ComponentType<any>,
    additionalMapper?: (state: ApplicationStateT) => Object | null = () => null,
): StatePropsT {
    const {id, name} = fieldDef;
    const {store} = state;
    const storeField = store[id];

    if (!storeField) {
        return {};
    }

    const {local: {scrollTo, pending = 0}} = state;
    const isScrollTarget = scrollTo === id;

    return assign(
        {
            Component,
            disabled: pending > 0,
            displayError: storeField.$$error,
            focused: storeField.$$focus,
            displayName: name,
            hint: storeField.$$hint || '',
            isScrollTarget,
            name: `field-${id}`,
            value: storeField.$$value || '',
            visible: storeField.$$visible,
        },
        additionalMapper(state),
    );
}

function _baseMapDispatchToProps(
    dispatch: Dispatch<*>,
    fieldDef: FieldDefT,
    additionalMapper?: (dispatch: Dispatch<*>) => Object | null = () => null,
): DispatchPropsT {
    const {id} = fieldDef;

    return assign(
        {
            onBlur: () => dispatch(fieldBlur(id)),
            onChange: (event, value) => dispatch(fieldChange(id, value.value)),
            onFocus: () => dispatch(fieldFocus(id)),
        },
        additionalMapper(dispatch),
    );
}

function _createWithComponent(
    fieldDef: FieldDefT,
    Component: React.ComponentType<any>,
    additionalStateMapper?: (state: ApplicationStateT) => Object,
    additionalDispatchMapper?: (dispatch: Dispatch<*>) => Object,
): React.Element<*> {
    const {id} = fieldDef;

    const stateMapper = state => _baseMapStateToProps(state, fieldDef, Component, additionalStateMapper);
    const dispatchMapper = dispatch => _baseMapDispatchToProps(dispatch, fieldDef, additionalDispatchMapper);

    const FieldComponent = connect(
        stateMapper,
        dispatchMapper,
    )(FormField);

    return <FieldComponent key={id} />;
}

export function createField(fieldDef: FieldDefT): React.Element<*> | null {
    const {type} = fieldDef;

    if (['input', 'swift', 'inn_ph', 'inn_ur', 'ogrnip'].includes(type)) {
        return createInput(fieldDef);
    } else if (type === 'textarea') {
        return createTextarea(fieldDef);
    } else if (type === 'iban_or_account') {
        return createIbanOrAccount(fieldDef);
    } else if (type === 'boolean') {
        return createCheck(fieldDef);
    } else if (type === 'date') {
        return createDate(fieldDef);
    } else if (type === 'date_18') {
        return createDate18(fieldDef);
    } else if (type === 'boolean_true') {
        return createCheckTrue(fieldDef);
    } else if (type === 'select') {
        return createSelect(fieldDef);
    } else if (type === 'radio') {
        return createRadio(fieldDef);
    } else if (type === 'oferta_agreement') {
        return createOfferCheck(fieldDef);
    } else if (type === 'attachments') {
        return createFileUpload(fieldDef);
    } else if (type === 'fias_address') {
        return createAddress(fieldDef);
    } else if (type === 'fias_ur_address') {
        return createAddress(fieldDef, true);
    } else if (type === 'text') {
        return createText(fieldDef);
    } else if (type === 'gfm_text') {
        return createGfmText(fieldDef);
    } else if (type === 'yandex_money_details') {
        return createYandexMoneyDetails(fieldDef);
    } else if (type === 'bik_with_account_or_yandex_money') {
        return createBikWithAccountOrYandexMoney(fieldDef);
    } else if (type === 'russian_passport_issuer_code') {
        return createRussianPassportIssuerCode(fieldDef);
    } else if (type === 'pfr') {
        return createPfrInput(fieldDef);
    } else if (type === 'campaign_creation') {
        return createCampaignCreation(fieldDef);
    } else if (type === 'mobile_app_creation') {
        return createMobileAppCreation(fieldDef);
    } else if (type === 'adfox_account') {
        return createAdfoxAccount(fieldDef);
    }

    return null;
}

function createInput(fieldDef: FieldDefT): React.Element<*> {
    return _createWithComponent(fieldDef, Input);
}

function createSelect(fieldDef: FieldDefT): React.Element<*> {
    const {dictionary = []} = fieldDef;
    const options = dictionary.map(({description, value}) => ({
        label: description,
        value,
    }));

    return _createWithComponent(fieldDef, Select, () => ({options}));
}

function createTextarea(fieldDef: FieldDefT): React.Element<*> {
    return _createWithComponent(fieldDef, Textarea);
}

function createIbanOrAccount(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        IbanOrAccount,
        state => ({
            value: state.store[id].$$value || {mode: 'iban', value: ''},
        }),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createDate(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        DatePicker,
        state => ({
            date: state.store[id].$$value,
            language: state.remote.user && state.remote.user.language,
        }),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createDate18(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        DatePicker,
        state => ({
            date: state.store[id].$$value,
            language: state.remote.user && state.remote.user.language,
            olderThan18: true,
        }),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createCheck(fieldDef: FieldDefT): React.Element<*> {
    const {id, name, value: defaultValue} = fieldDef;

    return _createWithComponent(
        fieldDef,
        Check,
        state => {
            const {$$value: storeValue} = state.store[id];
            const value = storeValue === null
                ? defaultValue || false
                : storeValue;

            return {
                checked: value,
                displayName: null,
                label: name,
                size: 'l',
                value: '',
            };
        },
        dispatch => ({
            onChange: (event, value) => {
                dispatch(fieldFocus(id));
                dispatch(fieldChange(id, value.checked));
                dispatch(fieldBlur(id));
            },
            onBlur: () => null,
            onFocus: () => null,
        }),
    );
}

function createCheckTrue(fieldDef: FieldDefT): React.Element<*> {
    const {id, name, value: defaultValue} = fieldDef;

    return _createWithComponent(
        fieldDef,
        Check,
        state => {
            const {$$value: storeValue} = state.store[id];
            const value = storeValue === null
                ? defaultValue || false
                : storeValue;

            return {
                checked: value,
                displayName: null,
                label: name,
                size: 'l',
                value: '',
            };
        },
        dispatch => ({
            onChange: (event, value) => {
                dispatch(fieldFocus(id));
                dispatch(fieldChange(id, value.checked));
                dispatch(fieldBlur(id));
            },
            onBlur: () => null,
            onFocus: () => null,
        }),
    );
}

function createRadio(fieldDef: FieldDefT): React.Element<*> {
    const {dictionary = []} = fieldDef;
    const options = dictionary.map(({description, value}) => ({
        label: description,
        value,
    }));

    return _createWithComponent(fieldDef, Radio, () => ({options, size: 'l', theme: 'common'}));
}

function createOfferCheck(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    function _mapStateToProps(state: ApplicationStateT): * {
        const message = __('I have read, agree and fully accept the [linkStart]terms of Offer[linkEnd], or confirm that I represent the interests of the person with whom the license agreement is concluded with respect to the results of intellectual activity');
        let offerUrl = '#';

        const {local: {countryId = '', branchId = ''}} = state;
        const {remote: {tree}} = state;

        if (tree && countryId && branchId) {
            const {branches} = tree[countryId];
            const branch = branches[branchId];
            offerUrl = branch.ofertaUrl;
        }

        return {
            checked: state.store[id].$$value || false,
            displayName: null,
            message,
            offerUrl,
            size: 'l',
            linkSize: 'l',
            value: '',
        };
    }

    return _createWithComponent(
        fieldDef,
        OfferCheck,
        _mapStateToProps,
        dispatch => ({
            onChange: (event, value) => {
                dispatch(fieldFocus(id));
                dispatch(fieldChange(id, value.checked));
                dispatch(fieldBlur(id));
            },
            onBlur: () => null,
            onFocus: () => null,
        }),
    );
}

function createFileUpload(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        FileUpload,
        state => ({
            files: state.store[id].$$value || [],
        }),
        dispatch => ({
            onChange: file => dispatch(fileUploadFieldAdd(id, getUniqId(), file)),
            onDelete: fileId => dispatch(fileUploadFieldDelete(id, fileId)),
        }),
    );
}

function createAddress(fieldDef: FieldDefT, isLegalAddress: boolean = false): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        Address,
        state => ({
            fias: state.remote.fias || {},
            isLegalAddress,
            pending: state.store[id].$$pending || false,
        }),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
            onFetchFias: guid => dispatch(fetchFias(guid, id)),
        }),
    );
}

function createText(fieldDef: FieldDefT): React.Element<*> {
    const {name} = fieldDef;

    return _createWithComponent(
        fieldDef,
        Text,
        () => ({
            displayName: null,
            text: name,
        }),
        () => ({
            onBlur: () => null,
            onFocus: () => null,
        }),
    );
}

function createGfmText(fieldDef: FieldDefT): React.Element<*> {
    const {name, style} = fieldDef;

    return _createWithComponent(
        fieldDef,
        GfmText,
        () => ({
            displayName: null,
            text: name,
            style,
        }),
        () => ({
            onBlur: () => null,
            onFocus: () => null,
        }),
    );
}

function createYandexMoneyDetails(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        YandexMoneyDetails,
        () => ({}),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createBikWithAccountOrYandexMoney(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        BikWithAccountOrYandexMoney,
        () => ({}),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createRussianPassportIssuerCode(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        withMask(Input, {mask: 'XXX-XXX', maskChar: 'X'}),
        () => ({}),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createPfrInput(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        withMask(Input, {mask: 'XXX-XXX-XXX XX', maskChar: 'X'}),
        () => ({}),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createCampaignCreation(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        CampaignCreation,
        () => ({}),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createMobileAppCreation(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        MobileAppCreation,
        () => ({}),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}

function createAdfoxAccount(fieldDef: FieldDefT): React.Element<*> {
    const {id} = fieldDef;

    return _createWithComponent(
        fieldDef,
        AdfoxAccount,
        state => ({
            pending: selectStoreFieldPending(state, id),
        }),
        dispatch => ({
            onChange: value => dispatch(fieldChange(id, value)),
        }),
    );
}
