// @flow
'use strict';

import {sortBy} from 'lodash';

import * as React from 'react';
import {connect} from 'react-redux';
import {Radio, Select} from 'teatime-components';

import {
    branchChange,
    countryChange,
} from '../../actions';

import css from './Fields.css';

import {createField} from '../FieldFactory/FieldFactory';
import {FormField} from '../../components/FormField/FormField';

import type {Dispatch} from 'redux';
import type {FieldDefT} from '../../types/fields';

import type {
    ApplicationStateT,
    BranchDataT,
    TreeDataT,
} from '../../types/state';

const VIP_COUNTRIES = [
    '225', // Russia
    '149', // Belarus
    '187', // Ukraine
    '159', // Kazakhstan
];

const COUNTRIES_SUB_HINTS = {
    '225': __('You will need to upload a scan of Russian passport, INN and SNILS'),
};

const SEARCH_EMPTY_TEXT = __('Result not found');

type StatePropsT = {
    branch: BranchDataT,
    branchId: ?string,
    countryId: ?string,
    createField: (field: FieldDefT) => *,
    disabled: boolean,
    showCountryAndBranch: boolean,
    tree: TreeDataT,
};

type DispatchPropsT = {
    onBranchChange: (branchId: string) => void,
    onCountryChange: (countryId: string) => void,
};

type PropsT = StatePropsT & DispatchPropsT;

class FieldGroup extends React.Component<{children?: React.Node}> {
    render(): React.Element<*> | null {
        const {children} = this.props;

        if (!children) {
            return null;
        }

        return (
            <div className={css.groupContainer}>
                <div className={css.groupBody}>
                    {children}
                </div>
                <div className={css.hr} />
            </div>
        );
    }
}

class Fields extends React.Component<PropsT> {
    constructor(props: PropsT) {
        super(props);

        this._onBranchChange = this._onBranchChange.bind(this);
        this._onCountryChange = this._onCountryChange.bind(this);
    }

    _onBranchChange: (event: Event, elem: Object) => void
    _onCountryChange: (event: Event, elem: Object) => void

    _onBranchChange(event: Event, elem: Object): void {
        const {onBranchChange} = this.props;

        onBranchChange(elem.value);
    }

    _onCountryChange(event: Event, elem: Object): void {
        const {onCountryChange} = this.props;

        onCountryChange(elem.value);
    }

    render(): React.Element<*> {
        const {showCountryAndBranch} = this.props;

        if (showCountryAndBranch) {
            return (
                <div>
                    <FieldGroup>
                        {this._renderHeader()}
                        {this._renderCountryField()}
                        {this._renderBranchesField()}
                    </FieldGroup>

                    {this._renderFields()}
                </div>
            );
        }

        return (
            <div>
                {this._renderFields()}
            </div>
        );
    }

    _renderHeader(): React.Element<*> {
        return (
            <h1 className={css.title}>{__('Partner registration form')}</h1>
        );
    }

    _renderCountryField(): React.Element<*> | null {
        const {
            countryId,
            disabled,
            tree,
        } = this.props;

        if (!tree) {
            return null;
        }

        const options = Object.keys(tree)
            .filter(countryId => !VIP_COUNTRIES.includes(countryId))
            .map(countryId => ({
                label: tree[countryId].countryName,
                value: countryId,
            }));

        const sortedOptions = sortBy(options, ['label', 'value']);

        const vipOptions = VIP_COUNTRIES
            .filter(countryId => tree.hasOwnProperty(countryId))
            .map(countryId => ({
                label: (tree[countryId] || {}).countryName,
                value: countryId,
            }));

        const allOptions = vipOptions.concat(
            [{label: '', type: 'option-group'}],
            sortedOptions,
        );

        const displaySubHint = countryId && COUNTRIES_SUB_HINTS.hasOwnProperty(countryId)
            ? COUNTRIES_SUB_HINTS[countryId]
            : undefined;

        return (
            <FormField
                displayName={__('Country')}
                displaySubHint={displaySubHint}
            >
                <Select
                    disabled={disabled}
                    name='country'
                    onChange={this._onCountryChange}
                    options={allOptions}
                    placeholder={__('Select a country')}
                    searchable={true}
                    searchEngine='includes'
                    searchEmptyText={SEARCH_EMPTY_TEXT}
                    size='l'
                    value={countryId}
                />
            </FormField>
        );
    }

    _renderBranchesField(): React.Element<*> | null {
        const {
            branchId,
            countryId,
            disabled,
            tree,
        } = this.props;

        if (!tree || !countryId) {
            return null;
        }

        const countryBranches = tree[countryId].branches;
        const {order: branchesOrder} = tree[countryId];

        const options = branchesOrder.map(branchId => ({
            label: countryBranches[branchId].branchName,
            value: branchId,
        }));

        return (
            <FormField displayName={__('Partnership form')}>
                <Radio
                    disabled={disabled}
                    name='branch'
                    onChange={this._onBranchChange}
                    options={options}
                    size='l'
                    value={branchId}
                />
            </FormField>
        );
    }

    _renderFields(): Array<React.Element<*>> | null {
        const {branch} = this.props;
        const {fields} = branch;

        if (fields.length <= 0) {
            return null;
        }

        const groups = getGroupedFields(fields);
        const renderedGroups = [];

        // eslint-disable-next-line guard-for-in
        for (const groupName in groups) {
            const renderedFields = groups[groupName].map(field => createField(field));

            renderedGroups.push(
                <FieldGroup key={groupName}>
                    <h1 className={css.title}>{groupName}</h1>
                    {renderedFields}
                </FieldGroup>
            );
        }

        return renderedGroups;
    }
}

function getGroupedFields(fields: Array<FieldDefT>): {[groupName: string]: Array<FieldDefT>} {
    const result = {};

    for (const field of fields) {
        const {groupName} = field;

        if (!result.hasOwnProperty(groupName)) {
            result[groupName] = [];
        }

        result[groupName].push(field);
    }

    return result;
}

const Container = connect(
    function mapStateToProps(state: ApplicationStateT): StatePropsT {
        const {
            remote: {tree = {}, user: {force} = {}},
            local: {pending = 0, branchId = '', countryId = ''},
        } = state;
        let {remote: {branch}} = state;

        if (!branch) {
            branch = {fields: []};
        }

        const {branchId: forceBranchId} = force || {};

        const showCountryAndBranch = !forceBranchId;

        return {
            branch,
            branchId,
            countryId,
            createField,
            disabled: pending > 0,
            showCountryAndBranch,
            tree,
        };
    },

    function mapDispatchToProps(dispatch: Dispatch<*>): DispatchPropsT {
        return {
            onBranchChange: (branchId: string) => {
                dispatch(branchChange(branchId));
            },
            onCountryChange: (countryId: string) => {
                dispatch(countryChange(countryId));
            },
        };
    },
)(Fields);

export {
    Container as Fields,
    VIP_COUNTRIES,
    SEARCH_EMPTY_TEXT,
};
