// @flow
'use strict';

import * as React from 'react';
import * as ReactDOM from 'react-dom';
import {Provider} from 'react-redux';
import {applyMiddleware, compose, createStore} from 'redux';
import createSagaMiddleware from 'redux-saga';
import Raven from 'raven-js';
import {fetchTree} from './actions';
import {Routes} from './routes.tsx';
import {initialState, rootReducer} from './reducers';
import rootSaga from './sagas';
import {composeWithScopedReducers} from './lib/reducers';

import {createBrowserHistory as createHistory} from 'history';
import {routerMiddleware as createRouterMiddleware, routerReducer} from 'react-router-redux';

import {configureRootTheme} from '@yandex-lego/components/Theme';
import {theme} from '@yandex-lego/components/Theme/presets/default/index';

import './index.css';
import './fonts.css';
import {simpleFormReducer} from './simple/reducers/simpleFormReducer.ts';
import {GLOSSARY} from './glossary.ts';

configureRootTheme({theme});

Raven
    .config(
        'https://f8ad4007d7bc4b559e947c6aecd9305e@sentry.iddqd.yandex.net/238',
        {release: process.env.LORDRAN_RELEASE || undefined},
    )
    .install();

const history = createHistory();
const routerMiddleware = createRouterMiddleware(history);

const sagaMiddleware = createSagaMiddleware();

const composeEnhancers = window.__REDUX_DEVTOOLS_EXTENSION_COMPOSE__ || compose;

const store = createStore(
    composeWithScopedReducers(
        rootReducer,
        {
            router: routerReducer,
            simpleForm: simpleFormReducer,
        },
    ),
    initialState,
    composeEnhancers(
        applyMiddleware(sagaMiddleware),
        applyMiddleware(routerMiddleware),
    ),
);

sagaMiddleware.run(rootSaga);

function render(): void {
    const root = document.getElementById('root');

    if (root === null) {
        throw new Error('Can\'t find root element.');
    }

    // Меняем title у страницы
    document.title = GLOSSARY.TITLE;

    ReactDOM.render((
        <Provider store={store}>
            <Routes history={history} />
        </Provider>
    ), root);
}

function initAndRender(): void {
    Raven.context(() => {
        store.dispatch(fetchTree());
        render();
    });
}

window.addEventListener('DOMContentLoaded', initAndRender);
