// @flow
'use strict';

const {assign} = Object;

let token;

/**
 * Returns promise that resolves into
 *     {ok: true, data: Object}
 * or
 *     {ok: false, error: string}
 *
 * @function
 * @param {string} url - url to fetch JSON from
 * @param {Object} opts - additional options to pass to fetch
 * @param {boolean} retryOnUnauthorized - retry with token on unauthorized
 * @returns {Promise}
 */
export function fetchJSON(url: string, opts: Object = {}, retryOnUnauthorized: boolean = false): Promise<Object> {
    return fetch(url, assign(
        {
            credentials: 'same-origin',
        },
        opts,
        {
            headers: assign(
                {},
                {Accept: 'application/json'},
                token ? {'X-Frontend-Authorization': token} : {},
                opts.headers,
            ),
        }
    ))
        .then(response => {
            if (response.ok) {
                return response.json().then(data => ({ok: true, data}));
            }

            const responseToken = response.headers.get('X-Frontend-Authorization');
            if (responseToken) {
                token = responseToken;
            }

            if (retryOnUnauthorized) {
                return fetchJSON(url, opts, false);
            }

            return response.json().then(data =>
                ({ok: false, data, error: `${response.status} ${response.statusText}`})
            );
        })
        .catch(error =>
            ({ok: false, error: error.message})
        );
}

/**
 * Returns promise that resolves into
 *     {ok: true, data: Object}
 * or
 *     {ok: false, error: string}
 *
 * @function
 * @param {string} url - url to fetch JSON from
 * @param {Object} data - JSON data to send
 * @param {boolean} retryOnUnauthorized - retry with token on unauthorized
 * @returns {Promise}
 */
export function postJSON(url: string, data: Object, retryOnUnauthorized: boolean = false): Promise<Object> {
    return fetchJSON(
        url,
        {
            body: JSON.stringify(data),
            cache: 'no-cache',
            method: 'POST',
            headers: {
                Accept: 'application/vnd.api+json',
                'Content-Type': 'application/vnd.api+json',
            },
        },
        retryOnUnauthorized,
    );
}

/**
 * Returns promise that resolves into
 *     {ok: true, data: Object}
 * or
 *     {ok: false, error: string}
 *
 * @function
 * @param {string} url - url to fetch JSON from
 * @param {File} file - File to send
 * @returns {Promise}
 */
export function uploadFile(url: string, file: File): Promise<Object> {
    const formData = new FormData();

    formData.append('upload', file);

    return fetchJSON(url, {
        body: formData,
        cache: 'no-cache',
        method: 'POST',
    });
}
