// @flow
'use strict';

import type {FieldGroupT, StoreFieldT, StoreT} from '../types/state';
import type {FieldDefT} from '../types/fields';

type GroupT = {
    name: string,
    checked: boolean,
};

type CampaignCreationValueT = {
    mode: 'site' | 'app',
    domain?: string,
    platform?: 'ios' | 'android',
    bundleId?: string,
    storeLink?: string,
}

type FieldDefaultsT = {[fieldId: string]: any};

const SKIP_FIELD_TYPES = ['text', 'gfm_text'];

function _isError(error: any): boolean {
    if (error === null) {
        return false;
    }

    if (Array.isArray(error)) {
        return error.reduce((acc, item) => acc || _isError(item), false);
    } else if (typeof error === 'object') {
        return Object.keys(error).length > 0;
    }

    return true;
}

function _isCampaignCreationFilled(field: StoreFieldT): boolean {
    if (!field) {
        return false;
    }

    const entries: Array<CampaignCreationValueT> = field.$$value;

    if (Array.isArray(entries) && entries.length === 0) {
        return false;
    }

    const firstEntry = entries[0];

    return Object.keys(firstEntry).length > 1;
}

function _fieldDefsById(fieldDefs: Array<FieldDefT>): {[fieldId: string]: FieldDefT} {
    return fieldDefs.reduce((acc, fieldDef) => {
        acc[fieldDef.id] = fieldDef;
        return acc;
    }, {});
}

export function areFieldsReady(fields: StoreT, fieldDefs: Array<FieldDefT>): boolean {
    const fieldDefsById = _fieldDefsById(fieldDefs);

    return Object.keys(fields).reduce(
        (acc, fieldId) => {
            const fieldDef = fieldDefsById[fieldId];

            if (!fieldDef) {
                return false;
            }

            const {type} = fieldDef;

            if (SKIP_FIELD_TYPES.includes(type)) {
                return acc;
            }

            const storeField = fields[fieldId];

            if (!storeField) {
                return false;
            }

            // если поле не видимо - не учитываем его в валидации
            if (!storeField.$$visible) {
                return true;
            }

            const campaignCreationFilled = type === 'campaign_creation'
                ? _isCampaignCreationFilled(storeField)
                : true;

            return acc &&
                storeField.$$touched &&
                !_isError(storeField.$$error) &&
                campaignCreationFilled &&
                !storeField.$$focus &&
                !storeField.$$pending;
        },
        true,
    );
}

export function haveUnskippableFields(fieldDefs: Array<FieldDefT>): boolean {
    const unskippableFields = fieldDefs.filter(({type}) => !SKIP_FIELD_TYPES.includes(type));

    return Boolean(unskippableFields.length);
}

export function getFilledGroups(
    fieldGroups: Array<FieldGroupT>,
    store: StoreT,
    fieldDefs: Array<FieldDefT>,
): Array<GroupT> {
    const filledGroups = fieldGroups.map(group => ({
        name: group.name,
        checked: areFieldsReady(
            group.fields.reduce(
                (acc, fieldId) => {
                    acc[fieldId] = store[fieldId];
                    return acc;
                },
                {}
            ),
            fieldDefs,
        ),
    }));

    return filledGroups;
}

export function getDefaultsByType(): FieldDefaultsT {
    return {
        attachments: [],
        campaign_creation: [{mode: 'site'}],
        mobile_app_creation: [{bundleId: '', platform: '', storeLink: ''}],
    };
}
