// @flow
'use strict';

import i18next from 'i18next';

import {getQueryLanguage} from '../lib/url';

export const LANGS = ['en', 'ru'];

export const DEFAULT_LANGUAGE = 'ru';

function initI18n() {
    const lang = getQueryLanguage() || DEFAULT_LANGUAGE;

    i18next.init({
        lng: lang,
        debug: false,
        initImmediate: false,
    });
}

export function addStrings(keysets: Object) {
    for (const lang of LANGS) {
        _addStringsToLang(lang, keysets[lang]);
    }
}

function _addStringsToLang(lang, keyset) {
    for (const keysetName of Object.keys(keyset)) {
        const updatedKeys = _formatPlurals(lang, keyset[keysetName]);
        i18next.addResources(lang, keysetName, updatedKeys);
    }
}

function _formatPlurals(lang, keys) {
    const result = {};

    for (const key of Object.keys(keys)) {
        const value = keys[key];

        if (Array.isArray(value)) {
            if (lang === 'ru') {
                result[`${key}_0`] = _formatParameters(value[0]);
                result[`${key}_1`] = _formatParameters(value[1]);
                result[`${key}_2`] = _formatParameters(value[2]);
            } else {
                result[key] = _formatParameters(value[0]);
                result[`${key}_plural`] = _formatParameters(value[1]);
            }
        } else {
            result[key] = _formatParameters(value);
        }
    }

    return result;
}

function _formatParameters(string) {
    return string.replace(/({|})/g, '$&$&');
}

export function translate(keyset: string, key: string, attrs?: Object): string {
    return i18next.t(`${keyset}:${key}`, attrs);
}

initI18n();
