// @flow
'use strict';

const {assign} = Object;

import {initialState} from '../reducers';

import type {ActionT} from '../actions';

import type {ApplicationStateT} from '../types/state';

import type {
    ComposableReducerT,
    SingleActionReducerT,
    SimpleReducerT,
} from '../reducers';

type ActionReducersT = {
    [actionType: string]: Array<SingleActionReducerT<*, *>>,
};

export function composeReducers(...reducers: Array<ComposableReducerT>): * {
    const actionReducers = reducers.reduce(
        (actionReducers: ActionReducersT, reducer: ComposableReducerT) => {
            for (const actionType of Object.keys(reducer)) {
                if (actionReducers.hasOwnProperty(actionType)) {
                    actionReducers[actionType].push(reducer[actionType]);
                } else {
                    actionReducers[actionType] = [reducer[actionType]];
                }
            }

            return actionReducers;
        },
        {},
    );

    function director(state: ApplicationStateT = initialState, action: ActionT<*, *>): ApplicationStateT {
        const {type} = action;
        const reducers = actionReducers[type] || [];

        return reducers.reduce(
            (state: ApplicationStateT, reducer: SingleActionReducerT<*, *>): ApplicationStateT =>
                reducer(state, action),
            state,
        );
    }

    return director;
}

export function composeWithScopedReducers(
    baseReducer: SimpleReducerT<*, *>,
    scopedReducers: {[scope: string]: SimpleReducerT<*, *>},
): * {
    function scopedDirector(state: ApplicationStateT = initialState, action: ActionT<*, *>): ApplicationStateT {
        let nextState = baseReducer(state, action);

        for (const scope of Object.keys(scopedReducers)) {
            const scopedState = state[scope];

            const nextScopedState = scopedReducers[scope](scopedState, action);

            nextState = assign({}, nextState, {[scope]: nextScopedState});
        }

        return nextState;
    }

    return scopedDirector;
}
