// @flow
'use strict';

const {assign} = Object;

import moment from 'moment';

import type {
    AddressT,
    FileDescT,
    MobileAppCreationT,
    AdfoxAccountT,
} from '../types/state';

import type {
    YandexMoneyDetailsT,
    BikWithAccountOrYandexMoneyT,
} from '../types/state';

const STREET_LEVEL = '7';
const ANDROID_PLATFORM_ID = 1;
const IOS_PLATFORM_ID = 2;

export const serializers = {
    input: (fieldValue: string | null): string | null => typeof fieldValue === 'string'
        ? fieldValue.trim()
        : null,

    iban_or_account: (fieldValue: {mode: 'iban' | 'account', value: string} | null): Object => {
        if (fieldValue === null) {
            return {iban: ''};
        }

        const {mode, value} = fieldValue;

        return mode === 'iban'
            ? {iban: value}
            : {account: value};
    },

    date: (fieldValue: Date | null): string | null => {
        if (fieldValue === null) {
            return null;
        }

        return moment(fieldValue).format('YYYY-MM-DD');
    },

    date_18: (fieldValue: Date | null): string | null => {
        if (fieldValue === null) {
            return null;
        }

        return moment(fieldValue).format('YYYY-MM-DD');
    },

    attachments: (fieldValue: Array<FileDescT> | null): Array<string> | null => {
        if (fieldValue === null) {
            return [];
        }

        return fieldValue.map(fileDesc => fileDesc.key);
    },

    text: (): * => null,
    gfm_text: (): * => null,
    campaign_creation: (): * => null,

    mobile_app_creation: (fieldValue: MobileAppCreationT): * => {
        const result = [];

        for (const entry of fieldValue) {
            const storeId = entry.bundleId;
            const storeUrl = entry.storeLink;
            const type = entry.platform === 'android' ? ANDROID_PLATFORM_ID : IOS_PLATFORM_ID;

            result.push({storeId, storeUrl, type});
        }

        return result;
    },

    fias_address: (fieldValue: AddressT): Object => {
        const {levels, house, apartment, building, structure, zipCode} = fieldValue;
        const guid = levels[STREET_LEVEL];

        return assign(
            {},
            {guid},
            house ? {house} : null,
            apartment ? {apartment} : null,
            building ? {building} : null,
            structure ? {structure} : null,
            zipCode ? {zip_code: zipCode} : null,
        );
    },

    fias_ur_address: (fieldValue: AddressT): Object => {
        const {levels, house, office, building, structure, zipCode} = fieldValue;
        const guid = levels[STREET_LEVEL];

        return assign(
            {},
            {guid},
            house ? {house} : null,
            office ? {office} : null,
            building ? {building} : null,
            structure ? {structure} : null,
            zipCode ? {zip_code: zipCode} : null,
        );
    },

    yandex_money_details: (fieldValue: YandexMoneyDetailsT): Object => {
        const {fields} = fieldValue;

        const {
            firstName,
            lastName,
            passportNumber,
            patronymicName,
            yandexMoney,
        } = (fields: any); // eslint-disable-line no-extra-parens

        return {
            first_name: firstName,
            last_name: lastName,
            passport_number: passportNumber,
            patronymic_name: patronymicName,
            yandex_money: yandexMoney,
        };
    },

    bik_with_account_or_yandex_money: (fieldValue: BikWithAccountOrYandexMoneyT): Object => {
        const {mode, fields} = fieldValue;

        if (mode === 'bank') {
            const {
                account,
                bik,
            } = (fields: any); // eslint-disable-line no-extra-parens

            return {
                account,
                bik,
            };
        }

        const {
            yandexMoney,
        } = (fields: any); // eslint-disable-line no-extra-parens

        return {
            yandex_money: yandexMoney,
        };
    },

    adfox_account: (fieldValue: AdfoxAccountT | null): Object => {
        const {
            adfoxOffer = false,
            hasAccount = false,
            hasAdfoxPaid = false,
            login = '',
            password = '',
        } = fieldValue || {};

        if (!hasAccount) {
            return {has_account: false};
        }

        return {
            adfox_offer: Boolean(adfoxOffer && hasAdfoxPaid),
            has_account: true,
            login,
            password,
        };
    },
};
