// @flow
'use strict';

import {
    camelize,
    decamelize,
} from 'humps';

import {getYandexTld} from './location';

export function camelizeObjectKeys(object: Object, deep: boolean = false): Object {
    return applyToObjectKeys(object, camelize, deep);
}

export function decamelizeObjectKeys(object: Object, deep: boolean = false): Object {
    return applyToObjectKeys(object, decamelize, deep);
}

function applyToObjectKeys<T1: Object, T2: Array<mixed>>(
    object: T1 | T2,
    func: string => string,
    deep: boolean = false
): $Shape<T1> | T2 {
    if (Array.isArray(object)) {
        return object.map(item => typeof item === 'object' && item !== null
            ? applyToObjectKeys(item, func, deep)
            : item
        );
    }

    if (typeof object === 'object' && object !== null) {
        const result = {};

        for (const key of Object.keys(object)) {
            const value = object[key];

            result[func(key)] =
                typeof value === 'object' && deep
                    ? applyToObjectKeys(value, func, true)
                    : value;
        }

        return result;
    }

    return object;
}

export function getOfferHeadings(html: string): Array<string> {
    const sectionRegExp = /<section .*? data-title="(?:\d+\.)?(.*?)".*?>/g;
    let execResults = [];
    const result = [];

    while ((execResults = sectionRegExp.exec(html)) !== null) {
        const heading = execResults[1];

        if (heading) {
            const trimmedHeading = heading.trim().toLowerCase();
            const capitalizedHeading = trimmedHeading.charAt(0).toUpperCase() + trimmedHeading.slice(1);

            result.push(capitalizedHeading);
        }
    }

    return result;
}

const UNIQ_ID_LENGTH = 8;
let uniqIdCounter = 0;

export function getUniqId(): string {
    uniqIdCounter++;

    const id = uniqIdCounter.toString();
    const padSize = UNIQ_ID_LENGTH - id.length;

    if (padSize < 0) { // Overflow
        uniqIdCounter = 0;
        return getUniqId();
    }

    const padding = '0'.repeat(padSize);

    return padding + id;
}

export function getSupportUrl(): string {
    let yandexTld = getYandexTld();

    // PI-28110
    const regex = RegExp(/.com$/i);
    yandexTld = yandexTld.replace(regex, '.ru');

    return `https://${yandexTld}/support/partner2/troubleshooting/joining.html`;
}
