import {SchemaT} from '../types/fields';

import {
    getMinLengthErrorMessage,
    getMaxLengthErrorMessage,
    getMinimalItemsCountError,
    GENERIC_ERROR_MESSAGE,
    EMPTY_VALUE_ERROR_MESSAGE,
} from './validators';

export type ValidationResultT = {
    ok: true,
} | {
    ok: false,
    error: string | {[subfieldId: string]: string},
};

export function validateSimpleJsonSchema(value: any, schema: SchemaT): ValidationResultT {
    const {
        enum: schemaEnum,
        maxLength,
        minLength,
        minItems,
        items,
        negativePattern,
        pattern,
        format,
        type,
    } = schema;

    if (type === 'boolean_true') {
        if (value === true) {
            return {ok: true};
        }

        return {ok: false, error: GENERIC_ERROR_MESSAGE};
    }

    if (type === 'boolean') {
        if (schemaEnum) {
            if (schemaEnum.includes(value)) {
                return {ok: true};
            }

            return {ok: false, error: EMPTY_VALUE_ERROR_MESSAGE};
        }

        return {ok: true};
    }

    if (type === 'array') {
        if (minItems && value.length < minItems) {
            return {ok: false, error: getMinimalItemsCountError(minItems)};
        }

        if (items) {
            const {enum: itemsAvailableValues, type: itemsType} = items;
            if (!itemsAvailableValues) {
                return {ok: true};
            }

            const found = value.find((oneValue: string) =>
                typeof oneValue !== itemsType || !itemsAvailableValues.includes(oneValue)
            );

            if (found) {
                return {ok: false, error: GENERIC_ERROR_MESSAGE};
            }
        }

        return {ok: true};
    }

    if (typeof value !== 'string') {
        return {ok: false, error: GENERIC_ERROR_MESSAGE};
    }

    const strval = value.trim();

    if (minLength && strval.length < minLength) {
        return {
            ok: false,
            error: minLength === 1 ? EMPTY_VALUE_ERROR_MESSAGE : getMinLengthErrorMessage(minLength)
        };
    }

    if (maxLength && strval.length > maxLength) {
        return {ok: false, error: getMaxLengthErrorMessage(maxLength)};
    }

    // если в jsonSchema нет условий minLength и maxLength, но в значении пустая строка
    if (strval === '') {
        return {ok: false, error: EMPTY_VALUE_ERROR_MESSAGE};
    }

    if (pattern) {
        const regexp = new RegExp(pattern);

        if (!regexp.test(strval)) {
            return {ok: false, error: GENERIC_ERROR_MESSAGE};
        }
    }

    if (negativePattern) {
        const patterns: Array<string> = typeof negativePattern === 'string'
            ? [negativePattern] : negativePattern;
        const RegExps = patterns.map(strRegExp => new RegExp(strRegExp));
        for (const regexp of RegExps) {
            if (regexp.test(strval)) {
                return {ok: false, error: GENERIC_ERROR_MESSAGE};
            }
        }
    }

    if (format === 'email') {
        if (!/^.+@.+\..+$/.test(strval)) {
            return {ok: false, error: GENERIC_ERROR_MESSAGE};
        }
    }

    return {ok: true};
}
