// @flow
'use strict';

import moment from 'moment';

import type {FieldDefT} from '../types/fields';
import type {StoreFieldT} from '../types/state';
import {validateSimpleJsonSchema} from './validateSimpleJsonSchema.ts';

export type ValidationResultT = {
    ok: true,
} | {
    ok: false,
    error: string | {[subfieldId: string]: string},
};

export type FieldMetaInfoT = {
    branchId: string;
};

export type RemoteValidationInfoT = {
    id: string,
    errorMessage: string,
    transformValue?: (value: any, meta: FieldMetaInfoT) => any,
    shouldValidateValue?: (value: any) => boolean,
    method?: 'GET' | 'POST',
}

export type ValidationInfoT = {
    remote: RemoteValidationInfoT | null,
};

export type RemoteErrorT = {
    errorToken: string | null,
    text: string | null,
};

export const GENERIC_ERROR_MESSAGE = __('Invalid value');
export const EMPTY_VALUE_ERROR_MESSAGE = __('Field must be filled');

export function getMinLengthErrorMessage(minLength: number): string {
    return __('Field should contain at least {minLength} characters', {count: minLength});
}

export function getMaxLengthErrorMessage(maxLength: number): string {
    return __('Field should contain at most {maxLength} characters', {count: maxLength});
}

export function getMinimalItemsCountError(minItemsCount: number): string {
    return __('It is necessary to select at least {minItemsCount} items', {count: minItemsCount});
}

const remoteValidationFields: {[fieldId: string]: RemoteValidationInfoT} = {
    swift: {
        id: 'swift',
        errorMessage: GENERIC_ERROR_MESSAGE,
    },
    iban_or_account: {
        id: 'iban',
        errorMessage: GENERIC_ERROR_MESSAGE,
        transformValue: value => value.value,
    },
    pfr: {
        id: 'pfr',
        errorMessage: GENERIC_ERROR_MESSAGE,
    },
    inn_ph: {
        id: 'inn_ph',
        errorMessage: GENERIC_ERROR_MESSAGE,
    },
    inn_ur: {
        id: 'inn_ur',
        errorMessage: GENERIC_ERROR_MESSAGE,
    },
    ogrnip: {
        id: 'ogrnip',
        errorMessage: GENERIC_ERROR_MESSAGE,
    },
    yandex_money_details: {
        id: 'swift',
        errorMessage: GENERIC_ERROR_MESSAGE,
        transformValue: value => value.fields.swift,
    },
    adfox_account: {
        id: 'adfox_account',
        errorMessage: __('Invalid credentials'),
        transformValue: ({adfoxOffer, hasAdfoxPaid, login, password}, {branchId}) => ({
            adfox_offer: Boolean(hasAdfoxPaid && adfoxOffer),
            branch_id: branchId,
            login,
            password,
        }),
        shouldValidateValue:
            ({ hasAccount, login, password }) => hasAccount && login !== undefined && password !== undefined,
        method: 'POST',
    },
};

export function getValidationInfo(field: StoreFieldT, fieldDef: FieldDefT): ValidationInfoT {
    const {id} = fieldDef;
    const validatorOpts = remoteValidationFields[id];
    const {$$value: value} = field;
    if (validatorOpts) {
        if (id === 'iban_or_account' && (!value || value.mode === 'account')) {
            return {remote: null};
        }

        if (id === 'yandex_money_details' && (!value || value.mode === 'yandexMoney')) {
            return {remote: null};
        }

        return {remote: validatorOpts};
    }

    return {remote: null};
}

export function validator(
    field: StoreFieldT,
    fieldDef: FieldDefT,
    remoteError: RemoteErrorT | null,
): ValidationResultT {
    const {$$touched: touched, $$value: value} = field;
    const {type} = fieldDef;

    const errorText = remoteError !== null ? remoteError.text : null;
    const errorToken = remoteError !== null ? remoteError.errorToken : null;

    if (!touched) {
        return {ok: true};
    }

    if (value === null) {
        return {ok: false, error: EMPTY_VALUE_ERROR_MESSAGE};
    }

    if (type === 'yandex_money_details') {
        return _validateYandexMoneyDetails(field, fieldDef, errorText);
    } else if (type === 'bik_with_account_or_yandex_money') {
        return _validateBikWithAccountOrYandexMoney(field);
    } else if (['fias_address', 'fias_ur_address'].includes(type)) {
        return _validateFiasAddress(field, fieldDef);
    } else if (type === 'campaign_creation') {
        return _validateCampaignCreation(field, fieldDef);
    } else if (type === 'mobile_app_creation') {
        return _validateMobileAppCreation(field, fieldDef);
    } else if (type === 'iban_or_account') {
        if (errorText) {
            return {ok: false, error: errorText};
        }

        return {ok: true};
    } else if (type === 'date_18') {
        if (value instanceof Date && moment().subtract(18, 'years').isAfter(value)) {
            return {ok: true};
        }

        return {ok: false, error: GENERIC_ERROR_MESSAGE};
    } else if (type === 'date') {
        if (value instanceof Date) {
            return {ok: true};
        }

        return {ok: false, error: GENERIC_ERROR_MESSAGE};
    } else if (type === 'adfox_account') {
        if (remoteError !== null && errorText && errorToken) {
            return {
                ok: false,
                error: {
                    text: errorText,
                    errorToken: String(errorToken),
                },
            };
        } else if (errorText) {
            return {ok: false, error: errorText};
        }

        return {ok: true};
    }

    if (errorText) {
        return {ok: false, error: errorText};
    }

    return validateSimpleJsonSchema(value, fieldDef.jsonSchema);
}

function _validateYandexMoneyDetails(
    field: StoreFieldT,
    fieldDef: FieldDefT,
    remoteError: string | null,
): ValidationResultT {
    const {$$value: value} = field;
    const errors = {};

    const schemas = {
        // Yandex.Money schemas
        firstName: {
            minLength: 1,
            type: 'string',
        },
        lastName: {
            minLength: 1,
            type: 'string',
        },
        passportNumber: {
            maxLength: 9,
            minLength: 8,
            pattern: '^([0-9]{9}|[А-Я]{2}[0-9]{6})$',
            type: 'string',
        },
        patronymicName: {
            minLength: 1,
            type: 'string',
        },
        yandexMoney: {
            type: 'string',
        },
    };

    if (remoteError) {
        errors.swift = remoteError;
    }

    const {fields} = value;

    const yandexMoneyFieldIds = ['firstName', 'lastName', 'passportNumber', 'patronymicName', 'yandexMoney'];

    const fieldsToCheck = yandexMoneyFieldIds;

    for (const fieldId of fieldsToCheck) {
        const result = validateSimpleJsonSchema(fields[fieldId], schemas[fieldId]);

        if (!result.ok) {
            const {error} = result;
            if (typeof error === 'string') {
                errors[fieldId] = error;
            } else {
                errors[fieldId] = GENERIC_ERROR_MESSAGE;
            }
        }
    }

    if (Object.keys(errors).length === 0) {
        return {ok: true};
    }

    return {ok: false, error: errors};
}

function _validateBikWithAccountOrYandexMoney(
    field: StoreFieldT,
): ValidationResultT {
    const {$$value: value} = field;
    const errors = {};

    const schemas = {
        // Bank schemas
        account: {
            type: 'string',
        },
        bik: {
            type: 'string',
        },

        // Yandex.Money schemas
        yandexMoney: {
            type: 'string',
        },
    };

    const {mode, fields} = value;

    const bankFieldIds = ['account', 'bik'];
    const yandexMoneyFieldIds = ['yandexMoney'];

    let fieldsToCheck = [];

    if (mode === 'bank') {
        fieldsToCheck = bankFieldIds;
    } else if (mode === 'yandexMoney') {
        fieldsToCheck = yandexMoneyFieldIds;
    }

    for (const fieldId of fieldsToCheck) {
        const result = validateSimpleJsonSchema(fields[fieldId], schemas[fieldId]);

        if (!result.ok) {
            const {error} = result;
            if (typeof error === 'string') {
                errors[fieldId] = error;
            } else {
                errors[fieldId] = GENERIC_ERROR_MESSAGE;
            }
        }
    }

    if (Object.keys(errors).length === 0) {
        return {ok: true};
    }

    return {ok: false, error: errors};
}

function _validateFiasAddress(
    field: StoreFieldT,
    fieldDef: FieldDefT,
): ValidationResultT {
    const {$$value: value} = field;
    const {type} = fieldDef;
    const errors = {};

    const schemas = {
        // Common address fields
        house: {
            minLength: 1,
            type: 'string',
        },
        zipCode: {
            maxLength: 20,
            pattern: '^[a-zA-Z0-9 ]+$',
            type: 'string',
        },

        // Legal address additional fields
        office: {
            minLength: 1,
            type: 'string',
        },
    };

    const fields = value;

    if (!value) {
        return {ok: false, error: GENERIC_ERROR_MESSAGE};
    }

    const fiasAddressFieldIds = ['house', 'zipCode'];
    const fiasUrAddressFieldIds = ['house', 'office', 'zipCode'];

    let fieldsToCheck = [];

    if (type === 'fias_address') {
        fieldsToCheck = fiasAddressFieldIds;
    } else if (type === 'fias_ur_address') {
        fieldsToCheck = fiasUrAddressFieldIds;
    }

    for (const fieldId of fieldsToCheck) {
        const result = validateSimpleJsonSchema(fields[fieldId], schemas[fieldId]);

        if (!result.ok) {
            const {error} = result;
            if (typeof error === 'string') {
                errors[fieldId] = error;
            } else {
                errors[fieldId] = GENERIC_ERROR_MESSAGE;
            }
        }
    }

    if (Object.keys(errors).length === 0) {
        return {ok: true};
    }

    return {ok: false, error: errors};
}

function _validateCampaignCreation(
    field: StoreFieldT,
    fieldDef: FieldDefT, // eslint-disable-line no-unused-vars
): ValidationResultT {
    const {$$value: entries} = field;

    const entryErrors = entries.map(_validateCampaignCreationEntry);

    if (entryErrors.length === 0) {
        return {ok: true};
    }

    return {ok: false, error: entryErrors};
}

function _validateCampaignCreationEntry(
    entry: Object,
): Object {
    const errors = {};

    const domainPartRe = '[^:\\s\\/.!@#$%^&*()\\[\\]\\{\\}\\?\\+;\\\'"`\\\\]+';
    const domainPattern =
        `^(?:https?:\\/\\/)?((?:${domainPartRe}\\.)+${domainPartRe})\\.?(:\\d{2,5})?($|\\/|\\?)`;
    const disallowDomainPattern = 'yandex\\.(?:(?:ru)|(?:com(?:\\.tr)?))$';

    const bundleIdPatterns = {
        android: '^[a-zA-Z][a-zA-Z0-9_]*(\\.[a-zA-Z][a-zA-Z0-9_]*)+$',
        ios: '^[a-zA-Z0-9][a-zA-Z0-9-]*(\\.[a-zA-Z0-9-]+)*$',
    };
    const bundleIdNegativePatterns = {
        android: '^com\\.example(?:\\.|$)',
        ios: ['^(?:public|dyn)(?:\\.|$)', '(?:^|\\.)(?:id\\d+)+(?:$|\\.)', '(?:^|\\.)(?:\\d|-)+(?:$|\\.)'],
    };

    const fields = entry;
    const {platform} = fields;

    const schemas = {
        // Site fields
        domain: {
            negativePattern: disallowDomainPattern,
            minLength: 1,
            pattern: domainPattern,
            type: 'string',
        },

        // App fields
        platform: {
            minLength: 1,
            pattern: '^(android|ios)$',
            type: 'string',
        },
        bundleId: {
            minLength: 1,
            negativePattern: bundleIdNegativePatterns[platform],
            pattern: bundleIdPatterns[platform],
            type: 'string',
        },
        storeLink: {
            minLength: 1,
            pattern: domainPattern,
            type: 'string',
        },
    };

    const {mode} = entry;

    const siteFieldIds = ['domain'];
    const appFieldIds = ['platform', 'bundleId', 'storeLink'];

    let fieldsToCheck = [];

    if (mode === 'site') {
        fieldsToCheck = siteFieldIds;
    } else if (mode === 'app') {
        fieldsToCheck = appFieldIds;
    }

    for (const fieldId of fieldsToCheck) {
        const result = validateSimpleJsonSchema(fields[fieldId], schemas[fieldId]);

        if (!result.ok) {
            const {error} = result;
            if (typeof error === 'string') {
                errors[fieldId] = error;
            } else {
                errors[fieldId] = GENERIC_ERROR_MESSAGE;
            }
        }
    }

    return errors;
}

function _validateMobileAppCreation(
    field: StoreFieldT,
    fieldDef: FieldDefT, // eslint-disable-line no-unused-vars
): ValidationResultT {
    const {$$value: entries} = field;

    const entryErrors = entries.map(_validateMobileAppCreationEntry);

    if (entryErrors.length === 0) {
        return {ok: true};
    }

    return {ok: false, error: entryErrors};
}

function _validateMobileAppCreationEntry(
    entry: Object,
): Object {
    const errors = {};

    const schemas = {
        platform: {
            minLength: 1,
            type: 'string',
        },
        bundleId: {
            minLength: 1,
            type: 'string',
        },
        storeLink: {
            minLength: 1,
            pattern: '^(http|https)://',
            type: 'string',
        },
    };

    const fields = entry;
    const fieldsToCheck = ['platform', 'bundleId', 'storeLink'];

    for (const fieldId of fieldsToCheck) {
        const result = validateSimpleJsonSchema(fields[fieldId], schemas[fieldId]);

        if (!result.ok) {
            const {error} = result;
            if (typeof error === 'string') {
                errors[fieldId] = error;
            } else {
                errors[fieldId] = GENERIC_ERROR_MESSAGE;
            }
        }
    }

    return errors;
}
