// @flow
'use strict';

import type {FieldDefT} from '../types/fields';

const {assign} = Object;

import type {
    FieldChangeActionT,
    FileUploadFieldAddActionT,
    FileUploadFieldDeleteActionT,
    FileUploadFieldFinishedActionT,
    InitStoreActionT,
    ReceiveBackendErrorsActionT,
    ScrollToActionT,
    SetFieldErrorActionT,
    SetFieldFocusActionT,
    SetFieldPendingActionT,
} from '../actions';
import {
    FIELD_CHANGE_ACTION,
    FILE_UPLOAD_FIELD_ADD_ACTION,
    FILE_UPLOAD_FIELD_DELETE_ACTION,
    FILE_UPLOAD_FIELD_FINISHED_ACTION,
    INIT_STORE_ACTION,
    RECEIVE_BACKEND_ERRORS_ACTION,
    SCROLL_TO_ACTION,
    SERIALIZE_FIELDS_ACTION,
    SET_FIELD_ERROR_ACTION,
    SET_FIELD_FOCUS_ACTION,
    SET_FIELD_PENDING_ACTION,
} from '../actions';

import {HAS_ADFOX_PAID_SERVICES_ERROR_TOKEN} from '../types/fields';

import {serializers} from '../lib/serializers';
import {evalCondition} from '../lib/evalCondition';

import type {ApplicationStateT, StoreFieldT, StoreT} from '../types/state';

function _updateStoreField(
    state: ApplicationStateT,
    id: string,
    values: $Shape<StoreFieldT>,
): ApplicationStateT {
    const {store} = state;
    const field = store[id];

    const nextStore = assign({}, store, {
        [id]: assign({}, field, values),
    });

    return assign({}, state, {store: nextStore});
}

/**
 * Remember received HAS_ADFOX_PAID_SERVICES token.
 *
 * @param {ApplicationStateT} state
 * @param {string} errorToken
 * @return {ApplicationStateT}
 */
function _receiveAdfoxAccountToken(
    state: ApplicationStateT,
    errorToken: string,
): ApplicationStateT {
    const {store} = state;
    const {adfox_account: adfoxAccount} = store;
    const {$$value} = adfoxAccount;
    const {hasAdfoxPaid} = $$value;

    const hasGotToken = errorToken === HAS_ADFOX_PAID_SERVICES_ERROR_TOKEN;

    if (!hasGotToken || hasAdfoxPaid) {
        return state;
    }

    return {
        ...state,
        store: {
            ...store,
            adfox_account: {
                ...adfoxAccount,
                $$value: {
                    ...$$value,
                    hasAdfoxPaid: hasGotToken,
                },
                $$error: {},
            },
        },
    };
}

export const fieldsReducer = {
    [FIELD_CHANGE_ACTION]: (state: ApplicationStateT, action: FieldChangeActionT): ApplicationStateT => {
        const {payload} = action;
        const {id, value} = payload;

        const nextState = _updateStoreField(state, id, {$$value: value, $$touched: true});
        const {store: nextStore, remote} = nextState;
        const {branch} = remote;

        if (!branch) {
            return nextState;
        }

        const {fields = []} = branch;

        // высчитываем видимость полей на основе уже составленного стора
        const finalStore = updateFieldsVisibility(nextStore, fields);

        return assign({}, state, {store: finalStore});
    },

    [INIT_STORE_ACTION]: (state: ApplicationStateT, action: InitStoreActionT): ApplicationStateT => {
        const {payload} = action;
        const {defaults} = payload;
        const {remote} = state;
        const {branch} = remote;

        if (!branch) {
            return assign({}, state, {store: {}});
        }

        const {fields = []} = branch;

        const nextStore: StoreT = {};

        for (const field of fields) {
            let value = null;
            let touched = false;

            if (field.hasOwnProperty('value')) {
                value = field.value;
                touched = true;
            } else if (defaults.hasOwnProperty(field.type)) {
                value = defaults[field.type];
            }

            nextStore[field.id] = {
                $$error: null,
                $$focus: false,
                $$hint: field.hint || null,
                $$pending: false,
                $$touched: touched,
                $$value: value,
                $$visible: true,
            };
        }

        // высчитываем видимость полей на основе уже составленного стора
        const finalStore = updateFieldsVisibility(nextStore, fields);

        return assign({}, state, {store: finalStore});
    },

    [SERIALIZE_FIELDS_ACTION]: (state: ApplicationStateT): ApplicationStateT => {
        const {local, remote: {branch = {}}, store} = state;

        if (!branch) {
            const serializedFields = {};
            const nextLocal = assign({}, local, {serializedFields});
            return assign({}, state, {local: nextLocal});
        }

        const {fields = []} = branch;

        const serializedFields = {};

        const fieldDefsById = fields.reduce((acc, value) => {
            acc[value.id] = value;
            return acc;
        }, {});

        for (const fieldId of Object.keys(store)) {
            const fieldDef = fieldDefsById[fieldId];
            const {$$value: value} = store[fieldId];

            const serializer = serializers[fieldDef.type];

            serializedFields[fieldId] = serializer
                ? serializer(value)
                : value;
        }

        const nextLocal = assign({}, local, {serializedFields});

        return assign({}, state, {local: nextLocal});
    },

    [SET_FIELD_ERROR_ACTION]: (state: ApplicationStateT, action: SetFieldErrorActionT): ApplicationStateT => {
        const {payload: {id, error}} = action;

        const errorToken = id === 'adfox_account' && error && typeof error !== 'string' && error.errorToken;

        const nextState = errorToken
            ? _receiveAdfoxAccountToken(state, errorToken)
            : state;

        return _updateStoreField(nextState, id, {$$error: error});
    },

    [SET_FIELD_FOCUS_ACTION]: (state: ApplicationStateT, action: SetFieldFocusActionT): ApplicationStateT => {
        const {payload: {id, focus}} = action;

        return _updateStoreField(state, id, {$$focus: focus, $$touched: true});
    },

    [SET_FIELD_PENDING_ACTION]: (state: ApplicationStateT, action: SetFieldPendingActionT): ApplicationStateT => {
        const {payload: {id, pending}} = action;

        return _updateStoreField(state, id, {$$pending: pending});
    },

    [RECEIVE_BACKEND_ERRORS_ACTION]:
        (state: ApplicationStateT, action: ReceiveBackendErrorsActionT): ApplicationStateT => {
            const {payload: {errors}} = action;
            const {store} = state;

            const storePatch = {};

            for (const fieldId of Object.keys(errors)) {
                if (!store.hasOwnProperty(fieldId)) {
                    continue;
                }

                const field = store[fieldId];
                const fieldErrors = errors[fieldId];
                const errorToShow = getErrorToShow(fieldId, fieldErrors);

                storePatch[fieldId] = assign({}, field, {$$error: errorToShow});
            }

            const nextStore = assign({}, store, storePatch);

            return assign({}, state, {store: nextStore});
        },

    [SCROLL_TO_ACTION]: (state: ApplicationStateT, action: ScrollToActionT): ApplicationStateT => {
        const {payload: {id}} = action;
        const {local} = state;

        const nextLocal = assign({}, local, {scrollTo: id});

        return assign({}, state, {local: nextLocal});
    },

    [FILE_UPLOAD_FIELD_ADD_ACTION]: (
        state: ApplicationStateT,
        action: FileUploadFieldAddActionT
    ): ApplicationStateT => {
        const {payload: {fieldId, fileId, file}} = action;
        const field = state.store[fieldId];

        const value = field.$$value || [];
        const newValue = value.concat({
            id: fileId,
            file,
            pending: true,
            ok: false,
        });

        return _updateStoreField(state, fieldId, {$$value: newValue, $$touched: true});
    },

    [FILE_UPLOAD_FIELD_DELETE_ACTION]: (
        state: ApplicationStateT,
        action: FileUploadFieldDeleteActionT
    ): ApplicationStateT => {
        const {payload: {fieldId, fileId}} = action;
        const field = state.store[fieldId];

        const value = field.$$value || [];
        const newValue = value.filter(item => item.id !== fileId);

        return _updateStoreField(state, fieldId, {$$value: newValue, $$touched: true});
    },

    [FILE_UPLOAD_FIELD_FINISHED_ACTION]: (
        state: ApplicationStateT,
        action: FileUploadFieldFinishedActionT
    ): ApplicationStateT => {
        const {payload: {fieldId, fileId, ok, key}} = action;
        const field = state.store[fieldId];

        const value = field.$$value;
        let newValue;

        if (!ok) {
            newValue = value.filter(item => item.id !== fileId);
        } else {
            newValue = value.map(
                item =>
                    item.id === fileId
                        ? assign({}, item, {pending: false, ok, key})
                        : item
            );
        }

        return _updateStoreField(state, fieldId, {$$value: newValue});
    },
};

function getErrorToShow(fieldId: string, fieldErrors: Array<Object>): Object | string {
    // Taking only first error
    const firstError = fieldErrors[0];
    const {description: errorDescription, error_token: errorToken} = firstError;

    if (fieldId === 'adfox_account') {
        // adfox_account field has complex layout with customized error rendering
        const text = errorDescription;
        return errorToken ? {text, errorToken} : {text};
    }

    return errorDescription;
}

function updateFieldsVisibility(store: StoreT, fields: Array<FieldDefT>): StoreT {
    const nextStore: StoreT = {};

    for (const field of fields) {
        const {dependencies} = field;

        nextStore[field.id] = {
            ...store[field.id],
            $$visible: dependencies ? evalCondition(nextStore, dependencies) : true,
        };
    }

    return nextStore;
}
