// @flow
'use strict';

import {mapValues, uniq} from 'lodash';

import {
    BRANCH_CHANGE_ACTION,
    COUNTRY_CHANGE_ACTION,
    RECEIVE_BRANCH_ACTION,
    RECEIVE_OFFER_ACTION,
    RECEIVE_OFFER_HEADINGS_ACTION,
    RECEIVE_TREE_ACTION,
    RECEIVE_USER_DATA_ACTION,
} from '../actions';

import {camelizeObjectKeys} from '../lib/utils';

const {assign} = Object;

import type {ApplicationStateT} from '../types/state';

import type {
    BranchChangeActionT,
    CountryChangeActionT,
    ReceiveBranchActionT,
    ReceiveOfferActionT,
    ReceiveOfferHeadingsActionT,
    ReceiveTreeActionT,
    ReceiveUserDataActionT,
} from '../actions';

export const treeReducer = {
    [BRANCH_CHANGE_ACTION]: (state: ApplicationStateT, action: BranchChangeActionT): ApplicationStateT => {
        const {local} = state;
        const {branchId} = action.payload;

        const nextLocal = assign({}, local, {branchId});

        return assign({}, state, {local: nextLocal});
    },

    [COUNTRY_CHANGE_ACTION]: (state: ApplicationStateT, action: CountryChangeActionT): ApplicationStateT => {
        const {local} = state;
        const {countryId} = action.payload;

        const nextLocal = assign({}, local, {countryId});

        return assign({}, state, {local: nextLocal});
    },

    [RECEIVE_BRANCH_ACTION]: (state: ApplicationStateT, action: ReceiveBranchActionT): ApplicationStateT => {
        const {local, remote} = state;
        const {payload} = action;

        // Clear branch and fieldGroups
        if (!payload) {
            const nextRemote = assign({}, remote, {branch: null});
            const nextLocal = assign({}, local, {fieldGroups: null});

            return assign({}, state, { remote: nextRemote, local: nextLocal });
        }

        const {data} = payload;

        const branch = camelizeObjectKeys(data, true);

        const groups = {};
        const orderedGroups = [];

        for (const fieldDef of branch.fields) {
            const {groupName, id} = fieldDef;

            if (!groups.hasOwnProperty(groupName)) {
                groups[groupName] = [];
            }

            groups[groupName].push(id);
            orderedGroups.push(groupName);
        }

        const uniqOrderedGroups = uniq(orderedGroups);

        const fieldGroups = uniqOrderedGroups.map(groupName => ({
            name: groupName,
            fields: groups[groupName],
        }));

        const nextRemote = assign({}, remote, {branch});
        const nextLocal = assign({}, local, {fieldGroups});

        return assign({}, state, {remote: nextRemote, local: nextLocal});
    },

    [RECEIVE_TREE_ACTION]: (state: ApplicationStateT, action: ReceiveTreeActionT): ApplicationStateT => {
        const {remote} = state;
        const {data} = action.payload;

        const tree = mapValues(data, country => {
            const camelizedCountry = camelizeObjectKeys(country);

            camelizedCountry.branches = mapValues(camelizedCountry.branches, branch =>
                camelizeObjectKeys(branch)
            );

            return camelizedCountry;
        });

        const nextRemote = assign({}, remote, {tree});

        return assign({}, state, {remote: nextRemote});
    },

    [RECEIVE_USER_DATA_ACTION]: (state: ApplicationStateT, action: ReceiveUserDataActionT): ApplicationStateT => {
        const {remote} = state;
        const {data} = action.payload;

        const nextRemote = assign({}, remote, {user: camelizeObjectKeys(data, true)});

        const {user} = nextRemote;
        if (user) {
            const {version} = user;

            if (version && typeof version === 'string') {
                // Если выставлять куку `form_version=2` то перл будет возвращать версию строкой
                const numberVersion = Number.parseInt(version, 10);
                nextRemote.user.version = Number.isNaN(numberVersion) ? 1 : numberVersion;
            }
        }

        return assign({}, state, {remote: nextRemote});
    },

    [RECEIVE_OFFER_ACTION]: (state: ApplicationStateT, action: ReceiveOfferActionT): ApplicationStateT => {
        const {remote} = state;
        const {data} = action.payload;

        const nextRemote = assign({}, remote, {offer: camelizeObjectKeys(data)});

        return assign({}, state, {remote: nextRemote});
    },

    [RECEIVE_OFFER_HEADINGS_ACTION]: (
        state: ApplicationStateT,
        action: ReceiveOfferHeadingsActionT,
    ): ApplicationStateT => {
        const {remote} = state;
        const {headings} = action.payload;

        const nextRemote = assign({}, remote, {headings});

        return assign({}, state, {remote: nextRemote});
    },
};
