// @flow
'use strict';

const {assign} = Object;

import {call, delay, select, put} from 'redux-saga/effects';
import Raven from 'raven-js';

import {fetchJSON, postJSON} from '../lib/fetch';

import {
    selectCurrentUserUrl,
    selectRestApiContextCampaignUrl,
    selectRestApiMobileAppSettingsUrl,
    selectRestApiMobileAppUrl,
    selectRestApiSiteUrl,
} from '../selectors/url';

import {selectStoreFieldByFieldDefType} from '../selectors/fields';

import {
    pushError,
} from '../actions';

const CAMPAIGN_CREATION_TYPE = 'campaign_creation';
const SITE_TYPE = 'site';
const CONTEXT_CAMPAIGN_TYPE = 'context_on_site_campaign';
const MOBILE_APP_TYPE = 'mobile_app';
const MOBILE_APP_SETTINGS_TYPE = 'mobile_app_settings';

const SITE_MODE = 'site';
const APP_MODE = 'app';

const SERVER_ERROR_MSG = __('Error creating campaigns');
const ERROR_DELAY = 1000;

const PLATFORMS = {
    android: 1,
    ios: 2,
};

export default function* campaignCreationSaga(): Generator<*, *, *> {
    const userUrl = yield select(selectCurrentUserUrl);
    const userResponse = yield call(fetchJSON, userUrl);

    if (!userResponse.ok) {
        Raven.captureMessage('User response not ok');
        yield put(pushError(new Error(SERVER_ERROR_MSG)));
        yield delay(ERROR_DELAY);
        return;
    }

    const {
        has_role_in_partner2: hasRole,
    } = userResponse.data;

    if (!hasRole) {
        Raven.captureMessage('User doesn\'t have role in partner2');
        yield put(pushError(new Error(SERVER_ERROR_MSG)));
        yield delay(ERROR_DELAY);
        return;
    }

    const campaignCreationField = yield select(selectStoreFieldByFieldDefType, CAMPAIGN_CREATION_TYPE);

    // Selected branch doesn't have campaign creation field
    if (campaignCreationField === null) {
        return;
    }

    const {$$value: entries} = campaignCreationField;

    if (!entries.length || entries.length <= 0) {
        Raven.captureMessage('No entries for campaign creation filled');
        yield put(pushError(new Error(SERVER_ERROR_MSG)));
        yield delay(ERROR_DELAY);
        return;
    }

    for (const entry of entries) {
        const {mode} = entry;

        if (mode === SITE_MODE) {
            const {domain, link, login, password} = entry;

            const siteUrl = yield select(selectRestApiSiteUrl);

            const sitePayload = {
                data: {
                    attributes: {
                        domain,
                        link_stat: link,
                    },
                    type: SITE_TYPE,
                },
            };

            if (login && password) {
                sitePayload.data.attributes = assign(
                    {},
                    sitePayload.data.attributes,
                    {
                        login,
                        pass: password,
                    },
                );
            }

            const siteResponse = yield call(postJSON, siteUrl, sitePayload, true);

            if (!siteResponse.ok) {
                Raven.captureMessage('Site creation error', {extra:
                    {
                        request: {url: siteUrl, payload: sitePayload},
                        response: {payload: siteResponse.data},
                    },
                });
                yield put(pushError(new Error(SERVER_ERROR_MSG)));
                yield delay(ERROR_DELAY);
                continue;
            }

            const {data: {id: siteId}} = siteResponse.data;

            const campaignUrl = yield select(selectRestApiContextCampaignUrl);

            const campaignPayload = {
                data: {
                    attributes: {
                        caption: __('Context campaign'),
                        domain_id: String(siteId),
                    },
                    type: CONTEXT_CAMPAIGN_TYPE,
                },
            };

            const campaignResponse = yield call(postJSON, campaignUrl, campaignPayload, true);

            if (!campaignResponse.ok) {
                Raven.captureMessage('Campaign creation error', {extra:
                    {
                        request: {url: campaignUrl, payload: campaignPayload},
                        response: {payload: campaignResponse.data},
                    },
                });
                yield put(pushError(new Error(SERVER_ERROR_MSG)));
                yield delay(ERROR_DELAY);
            }
        } else if (mode === APP_MODE) {
            const {platform, bundleId, storeLink} = entry;

            const siteUrl = yield select(selectRestApiMobileAppUrl);

            const sitePayload = {
                data: {
                    attributes: {
                        store_id: bundleId,
                        store_url: storeLink,
                        type: PLATFORMS[platform] || 1,
                    },
                    type: MOBILE_APP_TYPE,
                },
            };

            const siteResponse = yield call(postJSON, siteUrl, sitePayload, true);

            if (!siteResponse.ok) {
                Raven.captureMessage('Mobile app creation error', {extra:
                    {
                        request: {url: siteUrl, payload: sitePayload},
                        response: {payload: siteResponse.data},
                    },
                });
                yield put(pushError(new Error(SERVER_ERROR_MSG)));
                yield delay(ERROR_DELAY);
                continue;
            }

            const {data: {id: appId}} = siteResponse.data;

            const campaignUrl = yield select(selectRestApiMobileAppSettingsUrl);

            const campaignPayload = {
                data: {
                    attributes: {
                        app_id: String(appId),
                        caption: __('Mobile app campaign'),
                    },
                    type: MOBILE_APP_SETTINGS_TYPE,
                },
            };

            const campaignResponse = yield call(postJSON, campaignUrl, campaignPayload, true);

            if (!campaignResponse.ok) {
                Raven.captureMessage('Mobile app creation error', {extra:
                    {
                        request: {url: campaignUrl, payload: campaignPayload},
                        response: {payload: campaignResponse.data},
                    },
                });
                yield put(pushError(new Error(SERVER_ERROR_MSG)));
                yield delay(ERROR_DELAY);
            }
        }
    }
}
