// @flow
'use strict';

import {call, put, select} from 'redux-saga/effects';
import Raven from 'raven-js';
import {pick, mapKeys} from 'lodash';

import {fetchJSON} from '../lib/fetch';
import {getLocation, navigate} from '../lib/location';
import {
    getPassportRedirectUrl,
    getPostRegistrationUrl,
    shouldRedirectUser,
} from '../lib/redirects';

import {
    selectCurrentUserUrl,
    selectDashboardUrl,
    selectTreeUrl,
} from '../selectors/url';

import {
    selectSubpath,
} from '../selectors';

import {
    selectUserData,
} from '../selectors/remote.ts';

import {
    branchChange,
    countryChange,
    receiveTree,
    receiveUserData,
} from '../actions';

import type {UserDataT} from '../types/state';

import checkLanguageSaga from './checkLanguageSaga';

export default function* fetchTreeSaga(): Generator<*, *, *> {
    const userUrl = yield select(selectCurrentUserUrl);
    const userResponse = yield call(fetchJSON, userUrl);

    const project = yield select(selectSubpath);

    if (userResponse.ok) {
        yield put(receiveUserData({data: userResponse.data}));
    } else {
        throw new Error(`Failed to fetch user data: ${userResponse.error}`);
    }

    const userData = yield select(selectUserData);

    // User not logged in, redirecting to Passport
    if (!userData.isLogged) {
        const passportUrl = yield call(getPassportRedirectUrl);
        yield call(navigate, passportUrl);
        return;
    }

    Raven.setUserContext(Object.assign({id: userData.userId}, userData));
    const tagsData = pick(userData, [
        'canFillForm',
        'canonicalLogin',
        'displayLogin',
        'hasRoleInPartner2',
    ]);
    const prefixedTags = mapKeys(tagsData, (value, key) => `user.${key}`);
    Raven.setTagsContext(prefixedTags);

    const {
        force,
    } = userData;

    if (shouldRedirectUser(userData)) {
        yield call(redirectUser, userData);
        return;
    }

    yield call(checkLanguageSaga);

    const treeUrl = yield select(selectTreeUrl, project);
    const treeResponse = yield call(fetchJSON, treeUrl);

    if (treeResponse.ok) {
        yield put(receiveTree({data: treeResponse.data}));
    } else {
        throw new Error(`Failed to fetch branches: ${treeResponse.error}`);
    }

    if (force) {
        const {branchId, countryId} = force;

        yield put(countryChange(countryId));
        yield put(branchChange(branchId));
    }
}

function getUserRedirectUrl(userData: UserDataT): ?string {
    const {
        hasRoleInPartner2,
        canFillFormReason,
    } = userData;

    if (hasRoleInPartner2) {
        // Already registered in PI.
        // Redirecting to dashboard.

        const {pathname} = getLocation();

        // Should only be triggered during local development
        if (
            pathname.startsWith('/v2/dashboard') ||
            pathname.startsWith('/v2/mobile-mediation')
        ) {
            return;
        }

        return selectDashboardUrl();
    }

    if (canFillFormReason === 'need_postregistration') {
        return getPostRegistrationUrl();
    }
}

function* redirectUser(userData: UserDataT): Generator<*, *, *> {
    const url = yield call(getUserRedirectUrl, userData);
    if (url) {
        yield call(navigate, url);
    }
}
